package com.yandex.burp.extensions.plugins.audit;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IResponseInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.Utils;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;

public class FuzzOriginPlugin implements IAuditPlugin {

    private final int ISSUE_TYPE = 0x080a0020;
    private final String ISSUE_NAME = "Cross-Origin Resource Sharing Issue";
    private final String SEVERITY = "High";
    private final String CONFIDENCE = "Certain";
    private IBurpExtenderCallbacks callbacks;
    private IExtensionHelpers helpers;
    private HashSet<String> flags;
    private List<String> yandexLikeDomains = Arrays.asList("ya.com", "ya.ua", "yandex.net");

    public FuzzOriginPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
        this.flags = new HashSet<>();
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        /*
            TODO Check IScannerInsertionPoint Type
         */

        IHttpService httpService = baseRequestResponse.getHttpService();
        IRequestInfo req = helpers.analyzeRequest(baseRequestResponse.getRequest());
        if (req == null) return null;

        URL url = helpers.analyzeRequest(baseRequestResponse).getUrl();
        if (flags.contains(url.toString())) return null;
        else flags.add(url.toString());

        List<IScanIssue> issues = new ArrayList<>();
        String origin = new String();

        origin = "https://xyz.com";
        /*
            If response reflect https://xyz.com Origin return null, this is Public API
         */
        if (checkAccessControlAllowOriginHeader(baseRequestResponse, req, origin)) {
            return null;
        }

        origin = Utils.getHeaderValue(req, "origin");

        if (origin != null) {
            if (checkAccessControlAllowOriginHeader(baseRequestResponse, req, origin)) {
                List<IScanIssue> res = fuzzOriginHeader(baseRequestResponse, req, origin);
                if (res != null) issues.addAll(res);
            }
        }

        if (httpService.getPort() == 443 || httpService.getPort() == 80) {
            origin = httpService.getProtocol() + "://" + httpService.getHost(); // (http|https)://example.com
        } else {
            origin = httpService.getProtocol() + "://" + httpService.getHost()
                    + ":" + Integer.toString(httpService.getPort());
        }

        if (origin != null) {
            if (checkAccessControlAllowOriginHeader(baseRequestResponse, req, origin)) {
                List<IScanIssue> res = fuzzOriginHeader(baseRequestResponse, req, origin);
                if (res != null) issues.addAll(res);
            }
        }

        return issues.isEmpty() ? null : issues;
    }

    public boolean checkAccessControlAllowOriginHeader(
            IHttpRequestResponse baseRequestResponse,
            IRequestInfo req,
            String origin) {

        IHttpService httpService = baseRequestResponse.getHttpService();

        List<String> headers = Utils.setHeaderValue(req.getHeaders(), "Origin", origin);

        byte[] body = helpers.stringToBytes(helpers.bytesToString(
                baseRequestResponse.getRequest()).substring(req.getBodyOffset()));
        byte[] modifiedReq = helpers.buildHttpMessage(headers, body);
        IHttpRequestResponse checkHeadersRequestResponse = this.callbacks.makeHttpRequest(
                httpService,
                helpers.buildHttpMessage(headers, body));
        IResponseInfo resp = helpers.analyzeResponse(checkHeadersRequestResponse.getResponse());
        String accessControlAllowOriginHeader = Utils.getHeaderValue(resp, "Access-Control-Allow-Origin");

        if (accessControlAllowOriginHeader != null) {
            return accessControlAllowOriginHeader.equals(origin);
        }
        return false;
    }

    public List<IScanIssue> fuzzOriginHeader(
            IHttpRequestResponse baseRequestResponse,
            IRequestInfo req,
            String origin) {

        List<IScanIssue> issues = new ArrayList<>();
        List<String> evilOrigins = new ArrayList<>();
        String originProtocol = new String();
        String originHost = new String();
        String originHostWithoutTld = new String();
        String originHostWithoutTldReplaceDot = new String();
        String originTld = new String();
        int originPort = new Integer(-1);
        List<String> bypassList = Arrays.asList("-", "\"", "{", "}", "+", "_", "^", "%60", "!",
                "~", "`", ";", "|", "&", "(", ")", "*", ",", "$", "=", "+", "%0b");

        IHttpService httpService = baseRequestResponse.getHttpService();

        try {
            URL originUrl = new URL(origin);
            originProtocol = originUrl.getProtocol();
            originHost = originUrl.getHost();
            originPort = originUrl.getPort();
            int pos = originHost.lastIndexOf('.');
            originHostWithoutTld = originHost.substring(0, pos);
            originTld = originHost.substring(pos + 1);
            /*
            mail.example.com = > mail-example.com
            // TODO replace domains >2 levels a.b.c.example.com and manage "com.tr." and others 2 dots TLD
             */
            if (originHostWithoutTld.contains(".")) {
                originHostWithoutTldReplaceDot = originHostWithoutTld.replaceFirst("\\.", "-");
            }
        } catch (MalformedURLException e) {
            return null;
        }

        if (originPort == -1) {
            // (http|https)://example.com.evil.com
            evilOrigins.add(originProtocol + "://" + originHost + ".evil.com");
            // (http|https)://example.ruevil.com
            evilOrigins.add(originProtocol + "://" + originHost + "evil.com");
            // (http|https)://example.ru$evil.com
            for (String bypassChar : bypassList) {
                evilOrigins.add(originProtocol + "://" + originHost + bypassChar + ".evil.com");
            }
            // (http|https)://example.tatar
            evilOrigins.add(httpService.getProtocol() + "://" + originHostWithoutTld + ".tatar");
            // (http|https)://subdomain-example.com
            if (originHostWithoutTldReplaceDot != null && !originHostWithoutTldReplaceDot.isEmpty()) {
                evilOrigins.add(httpService.getProtocol() + "://" + originHostWithoutTldReplaceDot + "." + originTld);
            } else {
                // (http|https)://xuexample.com
                evilOrigins.add(originProtocol + "://" + "xu" + originHost);
            }

            for (String domain : yandexLikeDomains) {
                evilOrigins.add(originProtocol + "://" + domain);
            }
        } else {
            evilOrigins.add(originProtocol + "://" + originHost
                    + ".evil.com" + ":" + Integer.toString(originPort));
            evilOrigins.add(originProtocol + "://" + originHost
                    + "evil.com" + ":" + Integer.toString(originPort));
            for (String bypassChar : bypassList) {
                evilOrigins.add(originProtocol + "://" + originHost + bypassChar + ".evil.com"
                        + ":" + Integer.toString(originPort));
            }
            evilOrigins.add(originProtocol + "://" + originHostWithoutTld + ".tatar"
                    + ":" + Integer.toString(originPort));
            if (originHostWithoutTldReplaceDot != null && !originHostWithoutTldReplaceDot.isEmpty()) {
                evilOrigins.add(httpService.getProtocol() + "://" + originHostWithoutTldReplaceDot + "." + originTld
                        + ":" + Integer.toString(originPort));
            } else {
                evilOrigins.add(originProtocol + "://" + "xu" + originHost
                        + ":" + Integer.toString(originPort));
            }

            for (String domain : yandexLikeDomains) {
                evilOrigins.add(originProtocol + "://" + domain + ":" + originPort);
            }
        }

        for (String evilOrigin : evilOrigins) {
            List<String> headers = Utils.setHeaderValue(req.getHeaders(), "Origin", evilOrigin);
            byte[] body = helpers.stringToBytes(helpers.bytesToString(
                    baseRequestResponse.getRequest()).substring(req.getBodyOffset()));
            byte[] modifiedReq = helpers.buildHttpMessage(headers, body);
            IHttpRequestResponse attack = this.callbacks.makeHttpRequest(
                    httpService,
                    helpers.buildHttpMessage(headers, body));
            IScanIssue res = analyzeResponse(attack, evilOrigin);
            if (res != null) issues.add(res);
        }
        return issues.isEmpty() ? null : issues;
    }

    public IScanIssue analyzeResponse(IHttpRequestResponse requestResponse, String evilOrigin) {
        if (requestResponse.getResponse() == null) return null;
        IResponseInfo resp = helpers.analyzeResponse(requestResponse.getResponse());
        String accessControlAllowOriginHeader = Utils.getHeaderValue(
                resp,
                "Access-Control-Allow-Origin");
        String accessControlAllowCredentialsHeader = Utils.getHeaderValue(
                resp,
                "Access-Control-Allow-Credentials");

        if (accessControlAllowOriginHeader != null
                && accessControlAllowOriginHeader.equals(evilOrigin)
                && accessControlAllowCredentialsHeader != null
                && accessControlAllowCredentialsHeader.equals("true")) {
            List responseMarkers = new ArrayList(1);
            String i = "Access-Control-Allow-Origin";
            responseMarkers.add(new int[]{helpers.bytesToString(requestResponse.getResponse()).indexOf(i),
                    helpers.bytesToString(requestResponse.getResponse()).indexOf(i) + i.length()});
            String attackDetails = evilOrigin + " may be able to carry out privileged actions"
                    + " and retrieve sensitive information.\n\n"
                    + "https://portswigger.net/kb/issues/00200601_cross-origin-resource-sharing-arbitrary-origin-trusted";
            return new CustomScanIssue(requestResponse.getHttpService(),
                    this.helpers.analyzeRequest(requestResponse).getUrl(),
                    new IHttpRequestResponse[]{this.callbacks.applyMarkers(requestResponse, null, responseMarkers)},
                    attackDetails, ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                    "", "", "");
        }
        return null;
    }
}
