package com.yandex.burp.extensions.plugins.audit;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IRequestInfo;
import burp.IResponseInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;


/**
 * Created by procenkoeg on 03/12/2018.
 */
public class NginxOffBySlashPlugin implements IAuditPlugin {

    private final int ISSUE_TYPE = 0x080a0010;
    private final String ISSUE_NAME = "Nginx Off-by-Slash Molly";
    private final String SEVERITY = "High";
    private final String CONFIDENCE = "Certain";
    private IBurpExtenderCallbacks callbacks;
    private IExtensionHelpers helpers;
    private HashSet<String> flags;
    private List<String> payloads = new ArrayList<>();

    public NginxOffBySlashPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
        this.flags = new HashSet<>();
        this.initPayloads();
    }

    private void initPayloads(){
        payloads.add("/assets");
        payloads.add("/static");
        payloads.add("/images");
        payloads.add("/build");
        payloads.add("/img");
        payloads.add("/imgs");
        payloads.add("/css");
        payloads.add("/app");
        payloads.add("/_static");
        payloads.add("/admin");
        payloads.add("/dist");
        payloads.add("/fonts");
        payloads.add("/lib");
        payloads.add("/style");
        payloads.add("/styles");
        payloads.add("/stylesheets");
        payloads.add("/wp-content");
    }

    private String addDots(String str, int count) {
        String dots = new String(new char[count]).replace("\0", ".");
        return str + dots;
    }

    private IHttpRequestResponse makeRequestWithPayload(String payload, IHttpRequestResponse baseRequestResponse) {
        // get headers
        List<String> headers = helpers.analyzeRequest(baseRequestResponse).getHeaders();
        // add payload path
        headers.set(0, "GET " + payload + " HTTP/1.1");
        // build message
        byte[] attackBody = helpers.buildHttpMessage(headers, helpers.stringToBytes(""));
        // make http request
        IHttpRequestResponse attackRequestResponse = callbacks.makeHttpRequest(baseRequestResponse.getHttpService(),
                attackBody);
        // return
        return attackRequestResponse;
    }

    private IHttpRequestResponse tryPayload(String payload, IHttpRequestResponse baseRequestResponse) {
        short statusCode;
        String stagePayload;
        IHttpRequestResponse attackRequestResponse;

        // /static../ -> 403
        stagePayload = addDots(payload, 2) + "/";
        attackRequestResponse = makeRequestWithPayload(stagePayload, baseRequestResponse);
        statusCode = helpers.analyzeResponse(attackRequestResponse.getResponse()).getStatusCode();
        if (statusCode != 403) {
            return null;
        }

        // make 3 requests:
        // /static   -> 3xx
        // /static.  -> 3xx
        // /static.. -> 3xx
        for (int i = 0; i < 3; i++) {
            stagePayload = addDots(payload, i);
            attackRequestResponse = makeRequestWithPayload(stagePayload, baseRequestResponse);
            statusCode = helpers.analyzeResponse(attackRequestResponse.getResponse()).getStatusCode();
            if (!(statusCode >= 300 && statusCode < 400)) {
                return null;
            }
        }

        // /static... -> !3xx
        stagePayload = addDots(payload, 3);
        attackRequestResponse = makeRequestWithPayload(stagePayload, baseRequestResponse);
        statusCode = helpers.analyzeResponse(attackRequestResponse.getResponse()).getStatusCode();
        if ((statusCode >= 200 && statusCode < 300) || (statusCode >= 400 && statusCode < 500)) {
            return attackRequestResponse;
        }
        else {
            return null;
        }
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        IResponseInfo resp = helpers.analyzeResponse(baseRequestResponse.getResponse());
        IRequestInfo req = helpers.analyzeRequest(baseRequestResponse.getRequest());
        if (resp == null | req == null) return null;

        String hostUrl = baseRequestResponse.getHttpService().toString();

        // Scan host only once
        if (flags.contains(hostUrl)) return null;
        else flags.add(hostUrl);

        List<IScanIssue> issues = new ArrayList<>();

        for (String payload : payloads) {
            IHttpRequestResponse attackRequestResponse = tryPayload(payload, baseRequestResponse);
            if (attackRequestResponse != null) {

                URL vulnUrl = helpers.analyzeRequest(attackRequestResponse).getUrl();

                String issueDetails = "Incorrect Nginx configuration detected on host: \n<b>" + hostUrl + "</b>.\n" +
                        "Vulnerable url: <b>" + vulnUrl + "</b>.\n" +
                        "Read more about: https://i.blackhat.com/us-18/Wed-August-8/us-18-Orange-Tsai-Breaking-" +
                        "Parser-Logic-Take-Your-Path-Normalization-Off-And-Pop-0days-Out-2.pdf.\n" +
                        "Check nginx configs using Yodax: <b>https://wiki.yandex-team.ru/product-security/yodax/</b>.";

                issues.add(new CustomScanIssue(attackRequestResponse.getHttpService(), vulnUrl,
                        new IHttpRequestResponse[]{callbacks.applyMarkers(attackRequestResponse, null, null)},
                        issueDetails, ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                        "", "", ""));

                // break on first vulnerable payload
                break;
            }
        }

        return issues.isEmpty() ? null : issues;
    }
}
