package com.yandex.burp.extensions.plugins.audit;

import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;

public class PprofPlugin implements IAuditPlugin {

    private static final String SIGNATURE = "Types of profiles available";
    private static final int ISSUE_TYPE = 0x080a0067;
    private static final String ISSUE_NAME = "Pprof available";
    private static final String SEVERITY = "High";
    private static final String CONFIDENCE = "Certain";
    private static List<String> postfixes = Arrays.asList("debug/pprof/", "pprof/");
    private final IBurpExtenderCallbacks callbacks;
    private final IExtensionHelpers helpers;
    private final HashSet<String> checkedPaths;

    public PprofPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
        this.checkedPaths = new HashSet<>();
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        byte insertionPointType = insertionPoint.getInsertionPointType();

        // skip injections not in path
        if (insertionPointType == IScannerInsertionPoint.INS_URL_PATH_FILENAME
                || insertionPointType == IScannerInsertionPoint.INS_URL_PATH_FOLDER)
            return null;

        IHttpService httpService = baseRequestResponse.getHttpService();
        IRequestInfo request = this.helpers.analyzeRequest(httpService, baseRequestResponse.getRequest());

        URL requestUrl = request.getUrl();
        String path = normalizePath(requestUrl.getPath());
        String hostWithPath = requestUrl.getHost() + path;

        // skip file with extensions
        if (requestUrl.getPath().contains("."))
            return null;

        for (String postfix : postfixes) {
            if (checkedPaths.contains(hostWithPath + postfix))
                continue;

            IHttpRequestResponse testRequest = tryRequest(request, httpService, path, postfix);

            if (testRequest.getResponse() == null)
                continue;

            checkedPaths.add(hostWithPath + postfix);

            String body = helpers.bytesToString(testRequest.getResponse());

            if (body.contains(SIGNATURE))
                return generateIssues(request, testRequest);
        }

        return null;
    }

    private String normalizePath(String path) {
        if (!path.endsWith("/")) {
            return path + "/";
        }
        return path;
    }

    private List<IScanIssue> generateIssues(IRequestInfo request, IHttpRequestResponse testRequest) {
        List<IScanIssue> issues = new ArrayList<>();

        IRequestInfo testRequestParsed = this.helpers.analyzeRequest(testRequest);

        String attackDetails = "Pprof debug interface available at " +
                testRequestParsed.getUrl().toString();

        issues.add(new CustomScanIssue(
                testRequest.getHttpService(),
                request.getUrl(),
                new IHttpRequestResponse[]{
                        this.callbacks.applyMarkers(testRequest,
                                null,
                                null)
                },
                attackDetails,
                ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                "", "", "")
        );

        return issues;
    }

    private IHttpRequestResponse tryRequest(IRequestInfo request, IHttpService httpService, String path, String postfix) {
        String payloadPath = path + postfix;
        String requestLine = "GET " + payloadPath + " HTTP/1.1";

        List<String> headers = request.getHeaders();
        headers.set(0, requestLine);

        byte[] attackBody = helpers.buildHttpMessage(headers, helpers.stringToBytes(""));

        return this.callbacks.makeHttpRequest(httpService, attackBody);
    }
}
