package com.yandex.burp.extensions.plugins.audit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IResponseInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;


/**
 * Created by ilyaon on 08/07/2019.
 */
public class SSTIPlugin implements IAuditPlugin {

    private IBurpExtenderCallbacks callbacks;
    private IExtensionHelpers helpers;

    private static final int ISSUE_TYPE = 0x00101080;
    private static final String ISSUE_NAME = "Server Side Template Injection";
    private static final String SEVERITY = "High";
    private static final String CONFIDENCE = "Certain";

    // https://github.com/DiogoMRSilva/websitesVulnerableToSSTI/blob/master/javascript/dustjs/exploit.py#L32
    private static final String[] payloadTemplates = {
            "{@math key=\"%d\" method=\"multiply\" operand=\"%d\"/}", // DustJS
            "{{=%d*%d}}", // Dot
            "#{%d*%d}" // Slim
    };

    public SSTIPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfigd) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        IResponseInfo resp = helpers.analyzeResponse(baseRequestResponse.getResponse());
        IRequestInfo req = helpers.analyzeRequest(baseRequestResponse.getRequest());

        if (resp == null || req == null || resp.getStatusCode() != 200)
            return null;

        IHttpService httpService = baseRequestResponse.getHttpService();

        List<IScanIssue> issues = new ArrayList<>();
        if (isInputReflected(httpService, insertionPoint)) {
            issues.addAll(tryAttack(httpService, insertionPoint));
        }

        return issues;
    }

    private boolean isInputReflected(IHttpService httpService, IScannerInsertionPoint insertionPoint) {
        String marker = UUID.randomUUID().toString().replaceAll("-", "");

        IHttpRequestResponse request = this.callbacks.makeHttpRequest(httpService,
                insertionPoint.buildRequest(this.helpers.stringToBytes(marker)));

        if (request.getResponse() == null) return false;
        return helpers.bytesToString(request.getResponse()).contains(marker);
    }

    private List<IScanIssue> tryAttack(IHttpService httpService, IScannerInsertionPoint insertionPoint) {
        List<Integer> randomNumbers = Arrays.asList(
                ThreadLocalRandom.current().nextInt(100, 10_000),
                ThreadLocalRandom.current().nextInt(100, 10_000)
        );

        String randomHex = UUID.randomUUID().toString().replaceAll("-", "");

        String marker = randomHex.substring(0, 10)
                + randomNumbers.get(0) * randomNumbers.get(1)
                + randomHex.substring(10, 10);

        List<IScanIssue> issues = new ArrayList<>();
        for (String template : payloadTemplates) {
            String payload = randomHex.substring(0, 10)
                    + String.format(template, randomNumbers.get(0), randomNumbers.get(1))
                    + randomHex.substring(10, 10);


            IHttpRequestResponse attack = this.callbacks.makeHttpRequest(httpService,
                    insertionPoint.buildRequest(this.helpers.stringToBytes(payload)));

            if (attack.getResponse() == null) continue;
            String body = helpers.bytesToString(attack.getResponse());

            if (body.contains(marker)) {
                List<int[]> requestMarkers = Collections.singletonList(
                        insertionPoint.getPayloadOffsets(this.helpers.stringToBytes(payload))
                );

                List<int[]> responseMarkers = Collections.singletonList(new int[]{
                        body.indexOf(marker),
                        body.indexOf(marker) + marker.length()
                });

                String attackDetails = "Server Side Template Injection (https://portswigger.net/blog/server-side-template-injection)";

                issues.add(new CustomScanIssue(attack.getHttpService(),
                        this.helpers.analyzeRequest(attack).getUrl(),
                        new IHttpRequestResponse[] {
                                this.callbacks.applyMarkers(attack, requestMarkers, responseMarkers)},
                        attackDetails, ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                        "", "", ""));
            }
        }

        return issues;
    }
}
