package com.yandex.burp.extensions.plugins.grep;


import java.io.IOException;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashSet;
import java.util.List;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IResponseInfo;
import burp.IScanIssue;
import com.google.gson.Gson;
import com.google.gson.annotations.Expose;
import com.google.gson.annotations.SerializedName;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.Utils;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;
import okhttp3.FormBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;


import static com.yandex.burp.extensions.plugins.Utils.byteArrayToHex;


/**
 * Created by oxdef on 20.02.17.
 */
public class CspPlugin implements IGrepPlugin {

    private final int ISSUE_TYPE = 0x42000050;
    private final String ISSUE_NAME = "Content Security Policy related information";
    private final String SEVERITY_MEDIUM = "Medium";
    private final String SEVERITY_LOW = "Low";
    private final String CONFIDENCE = "Certain";
    private final OkHttpClient client = new OkHttpClient();
    private final Gson gson = new Gson();
    private IBurpExtenderCallbacks callbacks;
    private IExtensionHelpers helpers;
    private HashSet<String> flags;
    private String apiUrl = null;
    private List<String> skipHashes;

    public CspPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        helpers = callbacks.getHelpers();
        flags = new HashSet<>();

        if (extConfig.getCspPluginConfig() == null)
            throw new NullPointerException();

        apiUrl = extConfig.getCspPluginConfig().getApiUrl();
        skipHashes = extConfig.getCspPluginConfig().getsSkipHashes();
    }

    private static String escapeUnicode(String input) {
        StringBuilder b = new StringBuilder();
        final int length = input.length();
        for (int offset = 0; offset < length; ) {
            final int codepoint = input.codePointAt(offset);
            b.append(String.format("&#%d;", codepoint));
            offset += Character.charCount(codepoint);
        }
        return b.toString();
    }

    @Override
    public IScanIssue grep(IHttpRequestResponse baseRequestResponse) {
        String contentType = null;
        String policy = null;
        String issueDetails;
        String issueRemediation;
        short statusCode;
        IHttpService httpService = baseRequestResponse.getHttpService();
        IRequestInfo req = helpers.analyzeRequest(baseRequestResponse.getRequest());
        IResponseInfo resp = helpers.analyzeResponse(baseRequestResponse.getResponse());

        if (resp == null || req == null)
            return null;

        // Process only HTML responses with 200 status code
        contentType = Utils.getContentType(resp);
        statusCode = resp.getStatusCode();
        if (contentType == null || !contentType.equals("text/html") || statusCode != 200)
            return null;

        URL url = helpers.analyzeRequest(baseRequestResponse).getUrl();
        String originAndPath = url.getProtocol() + "://" + url.getHost() + ":" + url.getPort() + url.getPath();

        // Run once per origin + path
        if (flags.contains(originAndPath))
            return null;
        else
            flags.add(originAndPath);

        policy = getPolicyFromHeader(resp);

        if (policy != null) {

            try {
                String[] parts = policy.split(";");

                StringBuilder persistPolicy = new StringBuilder();
                for (String part : parts) {
                    String partTrim = part.trim();

                    if (partTrim.startsWith("report-uri")) {
                        continue;
                    }

                    if (partTrim.startsWith("script-src")) {
                        String[] srcs = partTrim.split(" ");
                        for (String source : srcs) {
                            if (source.startsWith("'nonce-")) {
                                continue;
                            }
                            persistPolicy.append(source);
                        }
                        continue;
                    }

                    persistPolicy.append(part);
                }

                MessageDigest message = MessageDigest.getInstance("MD5");
                message.update(persistPolicy.toString().getBytes());

                String hash = byteArrayToHex(message.digest());

                if (skipHashes != null) {
                    if (skipHashes.contains(hash)) {
                        return null;
                    }
                }
            } catch (NoSuchAlgorithmException e) {
                e.printStackTrace();
            }

            return analyzePolicy(policy, baseRequestResponse);
        } else {
            // Report empty CSP
            issueDetails = "The URL <b> " + url.toString() + "</b><br />" +
                    " returned an HTTP response without the strongly recommended Content Security Policy HTTP header";
            issueRemediation = "Please, read our recommendations (https://wiki.yandex-team.ru/product-security/csp/) and implement CSP in the service.";

            return new CustomScanIssue(httpService, url,
                    new IHttpRequestResponse[]{
                            this.callbacks.applyMarkers(baseRequestResponse, null, null)
                    },
                    issueDetails, ISSUE_TYPE, ISSUE_NAME, SEVERITY_MEDIUM, CONFIDENCE,
                    issueRemediation, "", "");
        }
    }

    private CustomScanIssue analyzePolicy(String policy, IHttpRequestResponse baseRequestResponse) {
        String body = null;
        Response response = null;
        CspTesterResult result = null;
        String issueDetails = null;
        String issueRemediation;

        RequestBody formBody = new FormBody.Builder()
                .add("policy", policy)
                .build();
        Request request = new Request.Builder()
                .url(apiUrl)
                .post(formBody)
                .build();

        try {

            response = client.newCall(request).execute();

            if (!response.isSuccessful())
                throw new IOException("Unexpected code " + response);

            String severity = SEVERITY_LOW;
            body = response.body().string();

            result = gson.fromJson(body, CspTesterResult.class);
            if (!result.issues.isEmpty()) {
                issueDetails = "There were identified incorrect or too permissive Content Security Policy <br />" +
                        "policies returned by the web application under analysis. The policy: <br />";
                issueDetails += "<pre>" + result.pretty + "</pre><br />Findings:<ul>";

                for (CspTesterIssue issue : result.issues) {
                    issueDetails += "\n<li>" + issue.desc + "</li>";
                    if (issue.severity != null && issue.severity.score >= 3) {
                        severity = SEVERITY_MEDIUM;
                    }
                }
                issueDetails += "</ul>";
                issueRemediation = "Please, read our recommendations (https://wiki.yandex-team.ru/product-security/csp/) and fix these issues.";
                return new CustomScanIssue(baseRequestResponse.getHttpService(),
                        helpers.analyzeRequest(baseRequestResponse).getUrl(),
                        new IHttpRequestResponse[]{
                                this.callbacks.applyMarkers(baseRequestResponse, null, null)
                        },
                        issueDetails, ISSUE_TYPE, ISSUE_NAME, severity, CONFIDENCE,
                        issueRemediation, "", "");
            }

        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    private String getPolicyFromHeader(IResponseInfo resp) {
        return Utils.getHeaderValue(resp, "Content-Security-Policy");
    }
}

class CspTesterIssueSeverity {

    @SerializedName("label")
    public String label;

    @SerializedName("score")
    public int score;
}

class CspTesterIssue {

    @SerializedName("desc")
    public String desc;

    @SerializedName("type")
    public String type;

    @SerializedName("directive")
    public String directive;

    @SerializedName("severity")
    public CspTesterIssueSeverity severity;
}

class CspTesterResult {

    @SerializedName("pretty")
    @Expose
    public String pretty;
    @SerializedName("issues")
    @Expose
    public List<CspTesterIssue> issues;
    @SerializedName("errors")
    @Expose
    public List<String> errors;
}
