#!/bin/env python
# -*- coding: utf-8 -*-
import argparse
import logging
import logging.handlers
import os
import sys

import sentry_sdk

from security.c3po.bin.load_plugins import load_plugins
from security.c3po.components.core.plugins import BasePlugin, AbstractPlugin
from security.c3po.components.core.util import (
    get_config,
)
from security.c3po.components.plugins.abc_reviewer.index import AbcReviewer  # noqa: F401
from security.c3po.components.plugins.blocker_concealer.index import BlockerConcealer  # noqa: F401
# noinspection PyUnresolvedReferences
from security.c3po.components.plugins.idm_checker.index import IdmChecker  # noqa: F401

sys.dont_write_bytecode = True


def plugin_main(plugin_name: str, config):
    os.environ["REQUESTS_CA_BUNDLE"] = config.get("common", "tls_ca_certs_bundle")

    plugins = list(filter(
        lambda p: p.__class__.__name__.lower() == plugin_name.lower(), [
            BlockerConcealer(config),
            AbcReviewer(config),
            IdmChecker(config)
        ]
    ))
    if len(plugins) > 1:
        raise Exception("Ambiguous modules found! Check your imports")
    if len(plugins) == 0:
        raise Exception("No modules found! Check your imports")
    plugin = plugins[0]
    try:
        plugin.start()
    except Exception as e:
        sentry_sdk.capture_exception(e)
        logging.error("Exception in plugin {}:".format(plugin.title))
        logging.exception(e)


def generic_main(config, verbosity):
    logging.getLogger("").setLevel(logging.NOTSET)

    log_path = config.get("common", "log_path")
    rotation_handler = logging.handlers.RotatingFileHandler(
        filename=log_path, backupCount=1, maxBytes=1 * 2 ** 30  # 1 Gb
    )
    logging.basicConfig(
        handlers=[rotation_handler, logging.StreamHandler(sys.stdout)],
        level=verbosity,
        format="%(asctime)s:%(module)s:%(name)s:%(levelname)s:%(message)s",
    )
    logging.info("Starting C3PO")

    # Fix YandexInternalCA support
    os.environ["REQUESTS_CA_BUNDLE"] = config.get("common", "tls_ca_certs_bundle")

    sentry_sdk.init(dsn=config.get("sentry", "dsn"))
    plugins = load_plugins(config, sentry_sdk)
    # meta_plugins = meta_load_plugins(config)
    # plugins += meta_plugins

    if config.getboolean("common", "dev_env"):
        sentry_logger = logging.getLogger("sentry.base.Client")
        sentry_logger.setLevel(logging.DEBUG)
        sentry_logger.addHandler(rotation_handler)

    for p in plugins:
        try:
            if isinstance(p, BasePlugin):
                p.main()
            elif isinstance(p, AbstractPlugin):
                p.start()
            else:
                logging.error(f"Unknown plugin type {type(p)}")
        except Exception as e:
            sentry_sdk.capture_exception(e)
            logging.error("Exception in plugin {}:".format(p.title))
            logging.exception(e)

    logging.info("C3PO shutdown")


def parse_args():
    parser = argparse.ArgumentParser(description="C3PO Robot")
    parser.add_argument(
        "-c", "--config", metavar="FILENAME", dest="config", help="Configuration file"
    )
    parser.add_argument(
        "-m", "--module", metavar="MODULE", dest="module", help="Run single module"
    )
    parser.add_argument(
        "-v", "--verbose", dest="verbose", action="store_true", help="Enable verbose logging"
    )
    args = parser.parse_args()
    return args, parser


def main():
    args, parser = parse_args()
    env = os.getenv("EXEC_ENV")

    if args.config:
        config_file = args.config
    elif env is not None and env.lower() == "prod":
        config_file = "prod_config.ini"
    else:
        config_file = "test_config.ini"
    config = get_config(config_file)

    verbosity = logging.INFO
    if args.verbose:
        verbosity = logging.DEBUG

    if args.module:
        plugin_main(args.module, config)
    elif args.config:
        generic_main(config, verbosity)
    else:
        parser.print_help()
        parser.exit(1)


if __name__ == "__main__":
    main()
