import logging
import os
from datetime import datetime, timedelta
from queue import PriorityQueue, Queue
from typing import Set, Dict, Optional, Union, Iterable, Tuple

from security.c3po.components.core.common import ABC_ID
from security.c3po.components.core.mappings import dep_to_duty, components_to_duty, Assignable, \
    AssignableDeps, duty_to_abc
from security.c3po.components.core.service import AbstractService, Features
from security.c3po.components.core.util import get_issue_author, safe_func, get_config
from startrek_client.collections import Issues

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class Manager(AbstractService):
    """
    Duties:
    vertis_everyday
    geo_everyday
    browser_everyday
    media_everyday
    bank_everyday
    zen_everyday
    market_everyday
    sdc_everyday
    taxi_everyday
    coresec_everyday
    duty-ticket (cloud)
    """
    _DUTY_REBUILD_TIME = timedelta(minutes=30)
    _FALLBACK_DUTY = 'coresec_everyday'

    def __init__(self, config):
        super(Manager, self).__init__(config)
        self._config = config
        self._handlers: Dict[str, PriorityQueue] = dict()
        self._boss_to_duty_departments: Dict[str, str] = dict()
        self._boss_to_duty_abc: Dict[str, str] = dict()

        self._duty_slug_to_login = dict()
        self._duty_login_to_slug = dict()
        self.__rebuild_duties(ABC_ID.SECURITY)
        # self.__generate_abc_mappings()
        self._build_duties_timestamp = dict()

        self._assignable_deps = AssignableDeps(
            self._abc, self._startrek
        )

    def requires(self) -> Features:
        return Features.ABC | Features.STARTREK | Features.STAFF

    """ BEGIN: PUBLIC API METHODS """

    def get_duty_slugs(self, abc_id: int = ABC_ID.SECURITY) -> Iterable[str]:
        """
        Returns list of all duty slugs
        """
        self._update_duties(abc_id)
        return self._duty_slug_to_login.get(abc_id).keys()

    def duty_slug_to_login(self, slug: str, abc_id: int = ABC_ID.SECURITY) -> Optional[str]:
        self._update_duties(abc_id)
        return self._duty_slug_to_login.get(abc_id).get(slug)

    def duty_login_to_slug(self, login: str, abc_id: int = ABC_ID.SECURITY) -> Optional[str]:
        self._update_duties(abc_id)
        return self._duty_login_to_slug.get(abc_id).get(login)

    @staticmethod
    def component_to_assignable(component_id: int, queue: str = None) -> Optional[Assignable]:
        if queue:
            scope = components_to_duty.get(queue, dict())
        else:
            scope = dict()
            for mapping in components_to_duty.values():
                scope.update(mapping)
        return scope.get(component_id)

    def resolve_duty(self, assignable: Optional[Union[Assignable, str]]) -> Optional[str]:
        """
        Resolves assignable object into staff login
        """
        if assignable is None or isinstance(assignable, str):
            return assignable
        return assignable.resolve_duty(self._assignable_deps)

    def resolve_group(self, assignable: Optional[Union[Assignable, str]]) -> Optional[str]:
        if assignable is None or isinstance(assignable, str):
            return assignable
        return assignable.resolve_group(self._assignable_deps)

    def issue_assignee(
        self,
        issue: Union[str, Issues],
        fallback: Assignable = None,
        process_components: bool = False,
    ) -> Optional[Assignable]:
        """
        :param fallback:
        :param process_components: When enabled performs component table lookup first
        :param issue: Startrek issue / Startrek ticket key
        :return: staff login, resolve strategy
        """
        if isinstance(issue, str):
            issue = self._startrek.issues[issue]

        if process_components:
            queue = issue.queue.key
            component_ids = list(map(lambda comp: int(comp.id), issue.components))
            q_comps = components_to_duty.get(queue)
            if q_comps:
                for comp_id in component_ids:
                    assignee = q_comps.get(comp_id)
                    if assignee:
                        return assignee

        author = get_issue_author(issue)
        return self.assignee_by_department(author, fallback)

    def resolve_issue_assignee(self, issue: Union[str, Issues], fallback: Union[Assignable, str] = None):
        queue = issue.queue.key
        if queue in ('SECTASK', 'DOSTUPREQ'):
            return self.resolve_duty(self.issue_assignee(issue, fallback))
        return self.resolve_group(self.issue_assignee(issue, fallback))

    def assignee_by_department(self, login: str, fallback_duty: Assignable = None) -> Optional[Assignable]:
        departments = self._staff.get_person_departments(login)
        if not departments:
            return fallback_duty
        dep_urls = list(map(lambda dep: dep.get('url'), departments))
        for dep in reversed(dep_urls):
            if dep in dep_to_duty:
                return dep_to_duty[dep]
        return fallback_duty

    def resolve_assignee_by_department(self, login: str, fallback_duty: Union[Assignable, str] = None) -> Optional[str]:
        return self.resolve_duty(self.assignee_by_department(login, fallback_duty))

    # def add_handler(
    #     self,
    #     handler: Callable[[Issues], Optional[str]],
    #     queue: str = None,
    #     priority: int = 5
    # ):
    #     if queue:
    #         handlers_queue = self._handlers.get(queue, PriorityQueue())
    #         handlers_queue.put((priority, handler))
    #     else:
    #         ...

    """ END: PUBLIC API METHODS """

    def _update_duties(self, abc_id: int):
        if (
            self._build_duties_timestamp.get(abc_id) is None or
            datetime.now() - self._build_duties_timestamp.get(abc_id) > Manager._DUTY_REBUILD_TIME
        ):
            self.__rebuild_duties(abc_id)
            self._build_duties_timestamp.update({abc_id: datetime.now()})

    def __rebuild_boss_to_duties(self):
        boss_to_duty: Dict[str, (str, int)] = dict()
        duty_abc_mapping = {
            'market_everyday': [905],  # Сервисы Маркета: https://abc.yandex-team.ru/services/meta_market/
            'browser_everyday': [4447]  # Суперапп (VS) https://abc.yandex-team.ru/services/meta_portalservices/
        }
        for duty_slug, abc_ids in duty_abc_mapping.items():
            for abc_id in abc_ids:
                boss = self._abc.get_service_head_by_id(abc_id)
                if not boss:
                    ...
                if boss in boss_to_duty:
                    ...  # rebuild with increased depth
                    del boss_to_duty[boss]
                else:
                    boss_to_duty.update({boss: duty_slug})

    @safe_func
    def __rebuild_duties(self, abc_id: int):
        duties = self._abc.get_shifts(abc_id)

        slug_to_login = self._duty_slug_to_login.get(abc_id, dict())
        login_to_slug = self._duty_login_to_slug.get(abc_id, dict())
        for duty in duties:
            person = duty['person']['login']
            duty_slug = duty['schedule']['slug']
            slug_to_login.update({duty_slug: person})
            login_to_slug.update({person: duty_slug})
        self._duty_slug_to_login.update({abc_id: slug_to_login})
        self._duty_login_to_slug.update({abc_id: login_to_slug})

        if abc_id != ABC_ID.SECURITY:
            return
        query = (
            'Queue: CLOUDDUTY '
            'Components: 47232 '
            'Status:!closed '
            '"Sort By": Created ASC'
        )
        issues = self._startrek.issues.find(query)

        for issue in issues:
            if issue.assignee:
                login = issue.assignee.login
                slug_to_login.update({'duty-ticket': login})
                login_to_slug.update({login: 'duty-ticket'})
                break

    def responsible(self, issue) -> Optional[str]:
        if type(issue) is str:
            issue = self._startrek.issues[issue]
        queue = issue.queue.key
        assignee = self.__process_handlers(queue, issue)
        if assignee:
            return assignee

        author = get_issue_author(issue)
        self.__creator_heads(author)

    def __creator_heads(self, author: str):
        bosses = self._staff.get_person_chief_list(author)
        for boss in reversed(bosses):
            if boss in self._boss_to_duty_departments:
                ...

    def __process_handlers(self, queue, issue):
        for handler in self._handlers.get(queue, []):
            try:
                assignee = handler(issue)
                if assignee:
                    return assignee
            except Exception as e:
                logging.error(e)

    def __generate_abc_mappings(
        self,
    ) -> Dict[str, Assignable]:
        process_queue: Queue[Tuple[int, Assignable]] = Queue()
        duty_mapping: Dict[Assignable, Set[str]] = dict()

        for duty_context, abc_ids in duty_to_abc.items():
            for abc_id in abc_ids:
                process_queue.put((abc_id, duty_context))

        collisions: Dict[str, int] = dict()
        while not process_queue.empty():
            abc_id, duty_context = process_queue.get()
            logger.debug('handling', abc_id, duty_context)
            boss = self._abc.get_service_head_by_id(abc_id)
            if boss is None:
                children = list(map(lambda child: child.get('id'), self._abc.get_service_children(abc_id)))
                logger.debug(f'abc_id={abc_id}: boss is none => adding children', children)
                for child_abc_id in children:
                    process_queue.put((child_abc_id, duty_context))
            elif boss in collisions:
                collision_abc = collisions.get('boss')
                if collision_abc:
                    process_queue.put((collision_abc, duty_context))
                    logger.debug(f'hit collision boss={boss} queue extended {collision_abc}')
            else:
                collisions.update({boss: abc_id})
                logins = duty_mapping.get(duty_context, set())
                logins.add(boss)
                duty_mapping.update({duty_context: logins})
            logger.debug('queue size', process_queue.qsize())

        logger.info('Built abc mapping:', duty_mapping)
        reverse_mapping = dict()
        for context, boss_set in duty_mapping.items():
            for boss in boss_set:
                reverse_mapping.update({boss: context})
        self._boss_to_duty_abc = reverse_mapping
        return reverse_mapping

    def sib(self) -> Set[str]:
        # Sync ABC https://abc.yandex-team.ru/services/security/
        return set(self._abc.get_people_by_id(ABC_ID.SECURITY, with_descendants=False))


if __name__ == '__main__':
    config = get_config(os.environ.get('CONFIG'))
    manager = Manager(config)
    # ...
