from __future__ import annotations

import random
from abc import abstractmethod
from dataclasses import dataclass
from datetime import datetime, timedelta
from typing import Callable, Tuple, Optional

from security.c3po.components.core.common import ABC_ID
from security.yaseclib.abc import Abc
from startrek_client import Startrek


@dataclass
class AssignableDeps:
    abc: Abc
    startrek: Startrek


@dataclass
class Assignable:
    obj: object = None
    iss_tag: str = None
    group: Assignable = None

    cacheable = False
    cache_time = timedelta(minutes=15)
    _ts: datetime = None
    _cached_val: Optional[str] = None

    def resolve_duty(self, deps: AssignableDeps) -> Optional[str]:
        if not self.cacheable:
            return self._resolve(deps)
        if self._ts is None or datetime.now() - self._ts > self.cache_time:
            self._cached_val = self._resolve(deps)
            self._ts = datetime.now()
        return self._cached_val

    def resolve_group(self, deps: AssignableDeps) -> Optional[str]:
        if self.group:
            return self.group.resolve_duty(deps)
        return self.resolve_duty(deps)

    @abstractmethod
    def _resolve(self, deps: AssignableDeps) -> Optional[str]:
        ...

    def __hash__(self) -> int:
        return hash(self.obj)


class CloudDuty(Assignable):
    obj: str = 'duty-ticket'

    def _resolve(self, deps: AssignableDeps) -> Optional[str]:
        query = (
            'Queue: CLOUDDUTY '
            'Components: 47232 '
            'Status:!closed '
            '"Sort By": Created ASC'
        )
        issues = deps.startrek.issues.find(query)

        for issue in issues:
            if issue.assignee:
                return issue.assignee.login


class Duty(Assignable):
    obj: str
    abc_id: int = ABC_ID.SECURITY

    def _resolve(self, deps: AssignableDeps) -> Optional[str]:
        duties = deps.abc.get_shifts(self.abc_id)
        for duty in duties:
            person = duty['person']['login']
            duty_slug = duty['schedule']['slug']
            if duty_slug == self.obj:
                return person


class Assignee(Assignable):
    obj: str

    def _resolve(self, deps) -> Optional[str]:
        return self.obj


class AssigneeFrom(Assignable):
    @staticmethod
    def default(lst):
        if lst:
            return random.choice(lst)

    obj: Tuple[str]
    strategy: Callable[[Tuple[str]], Optional[str]] = default

    def __init__(self, *objs: str):
        self.obj = objs

    def _resolve(self, deps) -> Optional[str]:
        return self.strategy(self.obj)


class Group:
    BANK = AssigneeFrom('dbeltukov', 'mprokopovich')
    TAXI = AssigneeFrom('audap', 'naumov-al', 'azizalimov')
    MARKET = AssigneeFrom('ichalykin', 'max-rogov', 'alferovri')
    MEDIA = AssigneeFrom('rshemyakin')
    FINTECH = AssigneeFrom('a-abakumov', 'ilyaon')
    EXPERIMENTS = AssigneeFrom('kaleda', 'pavkir', 'ezzer')
    SEARCH = AssigneeFrom('ilyaon', 'anton-k', 'melkikh', 'procenkoeg', 'ezaitov', 'a-abakumov', 'horus', 'shikari')
    BROWSER = AssigneeFrom('naumov-al')
    VERTIS = AssigneeFrom('ezzer', 'pavkir')
    ZEN = AssigneeFrom('pavkir', 'ezzer')
    GEO = AssigneeFrom('enr1g')
    SDC = AssigneeFrom('kaleda', 'ezzer', 'pavkir')
    CLOUD = AssigneeFrom('cheremushkin', 'axlodin', 'gpwn')
    CORESEC = AssigneeFrom(
        'melkikh', 'procenkoeg', 'ezaitov', 'a-abakumov', 'horus', 'shikari', 'ilyaon', 'anton-k'
    )


market_everyday = Duty('market_everyday', iss_tag='iss_market', group=Group.MARKET)
coresec_everyday = Duty('coresec_everyday', iss_tag='iss_search', group=Group.CORESEC)
zen_everyday = Duty('zen_everyday', iss_tag='iss_zen', group=Group.ZEN)
media_everyday = Duty('media_everyday', iss_tag='iss_media', group=Group.MEDIA)
taxi_everyday = Duty('taxi_everyday', iss_tag='iss_taxi', group=Group.TAXI)
sdc_everyday = Duty('sdc_everyday', iss_tag='iss_sdc', group=Group.SDC)
vertis_everyday = Duty('vertis_everyday', iss_tag='iss_vertis', group=Group.VERTIS)
geo_everyday = Duty('geo_everyday', iss_tag='iss_geo', group=Group.GEO)
browser_everyday = Duty('browser_everyday', iss_tag='iss_browser', group=Group.BROWSER)
bank_everyday = Duty('bank_everyday', iss_tag='iss_bank', group=Group.BANK)
cloud_duty = CloudDuty(iss_tag='iss_cloud', group=Group.CLOUD)

duty_to_abc = {
    market_everyday: {905},  # Сервисы Маркета: https://abc.yandex-team.ru/services/meta_market/
    browser_everyday: {4447},  # Суперапп (VS) https://abc.yandex-team.ru/services/meta_portalservices/
    taxi_everyday: {35915},  # MLU https://abc.yandex-team.ru/services/mlu/
    media_everyday: {861},  # Сервисы медиасервисов https://abc.yandex-team.ru/services/meta_media/
    coresec_everyday: {851},  # Сервисы Поискового портала https://abc.yandex-team.ru/services/meta_search/
    geo_everyday: {33186},  # Маршрутизация https://abc.yandex-team.ru/services/b2broutinggroup/
    zen_everyday: {1164}  # Сервисы Дзен https://abc.yandex-team.ru/services/discovery/
}

duty_to_deps = {  # https://staff.yandex-team.ru/departments/...
    market_everyday: {
        'yandex_mnt_security_0785_dep64681',
        'yandex_monetize_market',
        'outstaff_8244',  # Яндекс.Маркет Групп (Поддержка бизнеса)
        'ext_yamarket',  # Внешние консультанты Яндекс.Маркета
        'yandex_monetize_market_0677_dep13312',  # HR Маркета
    },
    zen_everyday: {
        'yandex_rkub_discovery',
        'outstaff_9036',  # Дзен (Поддержка бизнеса)
    },
    media_everyday: {
        'yandex_media',
        'outstaff_0206',  # Медиасервисы (Поддержка бизнеса)
        'outstaff_3210_2341_dep89011'  # Департамент Медиасервисы (Поддержка бизнеса)
    },
    bank_everyday: {
        'yandex_exp_dep92026_dep88005_dep85425_dep98906',  # Управление безопасности Банка
        'yandex_mnt_security_1327_dep33253',
    },
    Group.FINTECH: {
        'yandex_exp_dep92026',  # Финансовые сервисы Яндекса
        'outstaff_dep45883',  # Финансовый департамент (Поддержка бизнеса)
        'yandex_fin',  # Финансовый департамент
    },
    cloud_duty: {
        'yandex_exp_9053',
        'yandex_mnt_security_0785_dep81117',
    },
    taxi_everyday: {
        'yandex_rkub_taxi_dep80006',  # Райдтех
        'yandex_dep37494_dep60081',  # Фудтех
        'yandex_rkub_taxi_dep80055',  # Доставка
        'yandex_rkub_taxi_dep22593',  # Общие подразделения
        'yandex_mnt_security_0785_dep33396',
        'outstaff_2289',  # Такси (Поддержка бизнеса)
        'ext_dep08382',  # Внешние консультанты БЮ Такси
        'yandex_rkub_taxi_7119_1889_dep60401',  # HR Райдтеха
        'yandex_rkub_taxi_7119_1889_dep09447',  # HR Фудтеха
    },
    sdc_everyday: {
        'yandex_rkub_taxi_selfdriven',
        'outstaff_2289_dep91054',  # Беспилотные автомобили (Поддержка бизнеса)
        'yandex_rkub_taxi_7119_1889_dep21022'  # HR Беспилотных технологий
    },
    vertis_everyday: {
        'yandex_personal_vertserv',
        'yandex_mnt_security_0785_dep62737',
    },
    Duty('coresec_everyday', iss_tag='iss_search', group=Group.SEARCH): {
        'yandex_dep79124',  # Бизнес-группа Поиска и рекламных технологий
        'yandex_mnt_security_1327',
        'yandex_dep23271',  # Блок операционного директора
        'outstaff_3210',  # Поисковый портал (Поддержка бизнеса)
    },
    Duty('coresec_everyday', iss_tag='iss_hr'): {
        'yandex_edu',  # HR-департамент
    },
    geo_everyday: {
        'yandex_content',
        'outstaff_3676',  # Сервисы для малого бизнеса и Гео
    },
    browser_everyday: {
        'yandex_rkub_bromobtech',
    },
    # AssigneeFrom('nakorobko'): {
    #     'yandex_mnt_1500_3591_7987'
    # }
}
dep_to_duty = {
    dep: duty
    for duty, dep_set in duty_to_deps.items()
    for dep in dep_set
}
components_to_duty = {
    'SECTASK': {  # https://st-api.yandex-team.ru/v2/queues/SECTASK/components
        64073: browser_everyday,  # Браузер
        74524: coresec_everyday,  # Поисковый портал
        74525: Duty('coresec_everyday'),  # Общие подразделения
        74526: Duty('coresec_everyday', iss_tag='iss_hr'),  # HR
        88928: sdc_everyday,  # Беспилотные автомобили
        64072: geo_everyday,  # Геосервисы
        64075: market_everyday,
        64071: media_everyday,
        64076: taxi_everyday,
        64069: vertis_everyday,
        64070: zen_everyday,
        107844: Duty('bank_everyday')
    },
    'SECREVIEW': {  # https://st-api.yandex-team.ru/v2/queues/SECREVIEW/components
        57473: Group.TAXI,
        57683: AssigneeFrom('pavkir', 'ezzer'),
        57684: AssigneeFrom('pavkir', 'ezzer'),
        57685: Group.MARKET,
        57686: Assignee('rshemyakin'),
        57687: CloudDuty(),  # Cloud
        57688: AssigneeFrom('kaleda', 'pavkir', 'ezzer'),  # Experiments
        57689: AssigneeFrom('kaleda'),  # Other
        57693: AssigneeFrom('enr1g'),
        104195: Group.BANK,
        105257: AssigneeFrom('ilyaon'),  # FinOps
        105258: AssigneeFrom('ilyaon'),  # Pay
        105259: AssigneeFrom('a-abakumov'),  # Yandex 360
        87555: AssigneeFrom('a-abakumov'),  # Персональные сервисы
    },
    'SECAUDIT': {  # https://st-api.yandex-team.ru/v2/queues/SECAUDIT/components
        31293: browser_everyday,
        64060: zen_everyday,
        64059: vertis_everyday,
        64056: geo_everyday,
        64061: media_everyday,
        64055: market_everyday,
        64057: taxi_everyday,
        88935: sdc_everyday,
        104558: bank_everyday,
    }
}
