from __future__ import annotations

from abc import abstractmethod, ABC
from dataclasses import dataclass
from typing import Generic, TypeVar, Type, Tuple, Optional, get_origin, get_args, ValuesView


@dataclass
class C3POPluginInfo:
    name: str
    description: str
    maintainer: str

    @classmethod
    def __str__(cls) -> str:
        return cls.name


_Deps = TypeVar('_Deps')
_Config = TypeVar('_Config')
_State = TypeVar('_State')


class C3PORegistry:
    _plugins = dict()

    @classmethod
    def add_plugin(cls, plugin_cls):
        if not issubclass(plugin_cls, C3POPlugin):
            raise TypeError(f'Expected subclass of C3POPlugin, got {plugin_cls} instance')
        cls._plugins[plugin_cls.__name__] = plugin_cls

    @classmethod
    def find_plugin(cls, cls_name: str) -> Optional[C3POPlugin]:
        return cls._plugins.get(cls_name)

    @classmethod
    def get_plugins(cls) -> ValuesView[C3POPlugin]:
        return cls._plugins.values()


class C3POPlugin(ABC, Generic[_Deps, _Config, _State]):
    @staticmethod
    @abstractmethod
    def plugin_info() -> C3POPluginInfo:
        ...

    @staticmethod
    @abstractmethod
    def setup(dependencies: _Deps, config: _Config, previous_state: Optional[_State]) -> _State:
        ...

    @staticmethod
    @abstractmethod
    def run(dependencies: _Deps, config: _Config, state: _State) -> Optional[_State]:
        ...

    @classmethod
    def provide(
        cls,
        dependencies: _Deps
    ) -> 'C3POPluginWithDependencies[_Deps, _Config, _State]':
        return C3POPluginWithDependencies(
            plugin_type=cls,
            chosen_dependencies=dependencies
        )

    @classmethod
    def configure(
        cls,
        config: _Config
    ) -> 'C3POPluginWithConfig[_Deps, _Config, _State]':
        return C3POPluginWithConfig(
            plugin_type=cls,
            chosen_config=config
        )

    @classmethod
    def dependencies_type(cls) -> Type[_Deps]:
        return cls.__c3po_plugin_type_params()[0]

    @classmethod
    def config_type(cls) -> Type[_Config]:
        return cls.__c3po_plugin_type_params()[1]

    @classmethod
    def state_type(cls) -> Type[_State]:
        return cls.__c3po_plugin_type_params()[2]

    @classmethod
    def __c3po_plugin_type_params(cls) -> Tuple[Type[_Deps], Type[_Config], Type[_State]]:
        origin = None
        for base in cls.__orig_bases__:
            if get_origin(base) is C3POPlugin:
                origin = base
        deps_type, config_type, state_type = get_args(origin)
        return deps_type, config_type, state_type

    def __init_subclass__(cls, **kwargs):
        C3PORegistry.add_plugin(cls)


@dataclass
class _C3POPluginBuilder(Generic[_Deps, _Config, _State]):
    plugin_type: Type[C3POPlugin[_Deps, _Config, _State]]

    def plugin_info(self) -> C3POPluginInfo:
        return self.plugin_type.plugin_info()

    def dependencies_type(self) -> Type[_Deps]:
        return self.plugin_type.dependencies_type()

    def config_type(self) -> Type[_Config]:
        return self.plugin_type.config_type()

    def state_type(self) -> Type[_State]:
        return self.plugin_type.state_type()


@dataclass
class C3POPluginWithConfig(_C3POPluginBuilder[_Deps, _Config, _State]):
    chosen_config: _Config

    def provide(
        self,
        dependencies: _Deps
    ) -> 'C3POFullyConfiguredPlugin[_Deps, _Config, _State]':
        return C3POFullyConfiguredPlugin(
            plugin_type=self.plugin_type,
            chosen_dependencies=dependencies,
            chosen_config=self.chosen_config,
        )


@dataclass
class C3POPluginWithDependencies(_C3POPluginBuilder[_Deps, _Config, _State]):
    chosen_dependencies: _Deps

    def configure(
        self,
        config: _Config
    ) -> 'C3POFullyConfiguredPlugin[_Deps, _Config, _State]':
        return C3POFullyConfiguredPlugin(
            plugin_type=self.plugin_type,
            chosen_dependencies=self.chosen_dependencies,
            chosen_config=config,
        )


@dataclass
class C3POFullyConfiguredPlugin(_C3POPluginBuilder[_Deps, _Config, _State]):
    chosen_dependencies: _Deps
    chosen_config: _Config

    def setup(self, previous_state: Optional[_State] = None) -> 'C3POReadyToRunPlugin[_Deps, _Config, _State]':
        state = self.plugin_type.setup(self.chosen_dependencies, self.chosen_config, previous_state)
        return C3POReadyToRunPlugin(
            plugin_type=self.plugin_type,
            chosen_dependencies=self.chosen_dependencies,
            chosen_config=self.chosen_config,
            state=state
        )


@dataclass
class C3POReadyToRunPlugin(_C3POPluginBuilder[_Deps, _Config, _State]):
    chosen_dependencies: _Deps
    chosen_config: _Config
    state: _State

    def __call__(self, *args, **kwargs):
        return self.plugin_type.run(
            self.chosen_dependencies,
            self.chosen_config,
            self.state,
        )
