# import inspect
import logging
import os
from abc import abstractmethod
from typing import List

# import psutil as psutil
# from crontab import CronTab
from jinja2 import Environment, PackageLoader, FileSystemLoader
from pathlib import Path
from security.c3po.components.core.service import AbstractService, Features

DEFAULT_ENABLED = True
DEFAULT_MAINTAINER = "@ya-andrei"
RESOURCE_PATH = os.getenv('RESOURCE_PATH', '/c3po/resources')


def get_all_plugin_types() -> List[type]:
    return MetaPlugin.all_plugins


class MetaPlugin(type):
    all_plugins = []

    def __new__(cls, name, bases, namespaces):
        """
        Automatically generate READMEs for modules and collects plugins into static all_plugins field
        """
        global all_plugins
        cl = super().__new__(cls, name, bases, namespaces)

        #  if object not in bases:
        #      MetaPlugin.__gen_readme(
        #          os.path.dirname(os.path.abspath(inspect.getsourcefile(cl))),
        #          title=namespaces.get("title", name),
        #          description=namespaces.get("description", ""),
        #          enabled=namespaces.get("enabled", DEFAULT_ENABLED),
        #          requires=cl.requires(),
        #          maintainer=namespaces.get("maintainer", DEFAULT_MAINTAINER),
        #      )
        #      MetaPlugin.all_plugins.append(cl)
        return cl

    @staticmethod
    def __gen_readme(filepath, **kwargs):
        env = Environment(
            loader=PackageLoader("security.c3po.components.core", "templates")
        )
        template = env.get_template("README.md")
        rendered_outp = template.render(**kwargs)
        with open(os.path.join(filepath, "README.md"), "w") as f:
            f.write(rendered_outp)


# Yes, inherit from this class
class AbstractPlugin(AbstractService, metaclass=MetaPlugin):
    """
    Example:

    class MyPlugin(AbstractPlugin):
        title = "My plugin"
        description = "It does #1 #2 #3"

        def requires(self) -> Features:
            return Features.ABC | Features.IMPULSE

        def setup(self):
            self._abc.do_something()
            self._impulse.do_something_else()
            return dict(x=3, y=2)

        def main(self, ctx):
            print(f'''x is {ctx.get('x')} and y is {ctx.get('y')}''')

        def teardown(self):
            print('Done!')
    """

    # Fields allowed to set:
    enabled = DEFAULT_ENABLED
    # run_schedule = CronTab(user=psutil.Process().username())
    title = None  # Set title
    description = ""  # Write description
    maintainer = DEFAULT_MAINTAINER
    templates_path = None

    def __init__(self, config):
        """
        Sets commonly used fields and runs initial __setup process
        """
        super(AbstractPlugin, self).__init__(config)
        self._config = config

        self._env = None
        if self.templates_path:
            self._env = Environment(
                loader=FileSystemLoader(
                    self.resource_path() / self.templates_path
                )
            )

        # Parse commonly used groups of people from configs
        # self._officers = self.__config_getlist("plugins.sectask", "officers")
        # self._default_assignee = config.get("plugins.sectask", "default_assignee")
        # self._abuse_assignee = config.get("plugins.sectask", "abuse_assignee")
        # self._pastebin_assignee = config.get("plugins.sectask", "pastebin_assignee")
        # self._pki_code = config.getint("plugins.sectask", "pki_component")

        self.setup()

    def requires(self) -> Features:
        """
        Override this method in your child class to specify objects you would like to use.
        Return Features object
        """
        return Features.NONE

    @staticmethod
    def __gen_cron():
        raise NotImplementedError

    def resource_path(self):
        context = self.__class__.__name__
        filepath = Path(RESOURCE_PATH) / context
        if not filepath.exists():
            filepath.mkdir(parents=True)
        return filepath

    def __config_getlist(self, *conf_path: str, item_type: type = str) -> List:
        return [item_type(i.strip()) for i in self._config.get(*conf_path).split(",")]

    def start(self) -> bool:
        """
        Implements module lifecycle
        Do not override this method
        Returns True if plugin completed successfully
        Returns False if plugin is disabled or exited with exception
        """
        if not self.enabled:
            logging.debug(f"Plugin {type(self).__name__} is disabled")
            return False

        logging.info(f"Executing main {type(self).__name__}")
        self.main()
        logging.info(f"Executing teardown {type(self).__name__}")
        self.teardown()

    @abstractmethod
    def setup(self):
        """
        Called before main function
        Override method to set proper environment and initialise self context
        """

    @abstractmethod
    def main(self):
        """
        Implement your plugin functionality here
        """
        raise NotImplementedError

    @abstractmethod
    def teardown(self):
        """
        Called after main function
        Clean environment and free used resources
        """

    def __str__(self) -> str:
        return self.title or self.__name__

    def info(self) -> str:
        return "\n".join(
            (
                f"Plugin:\t{str(self)}",
                f"Requires:\t{self._require}" f"Description:\t{self.description}",
            )
        )


# WARNING: That is now legacy and treated as deprecated
class BasePlugin(object):
    config = None
    ptype = None
    title = "Base plugin"
    desc = "Some text about base plugin"
    sentry_module = None

    def __init__(self, config, sentry_module):
        self.config = config
        self.sentry_module = sentry_module

    def resource_path(self):
        context = self.__class__.__name__
        filepath = Path(RESOURCE_PATH) / context
        if not filepath.exists():
            filepath.mkdir(parents=True)
        return filepath

    def setup(self):
        """called before the plugin is asked to do anything"""
        raise NotImplementedError

    def main(self):
        """called when the plugin is asked to do anything"""
        raise NotImplementedError

    def teardown(self):
        """called to allow the plugin to free anything"""
        raise NotImplementedError

    def _config_getlist(self, section, option):
        return [i.strip() for i in self.config.get(section, option).split(",")]

    def _config_getintlist(self, section, option):
        return [int(i.strip()) for i in self.config.get(section, option).split(",")]
