from enum import Flag, auto, _decompose
from typing import Optional
from abc import abstractmethod
import logging

from startrek_client import Startrek
from security.yaseclib.abc import Abc
from security.yaseclib.gap import Gap
from security.yaseclib.idm2 import Idm
from security.yaseclib.impulse import ImpulseQueuer
from security.yaseclib.moe import Moe
from security.yaseclib.puncher import Puncher
from security.yaseclib.staff import Staff
from security.yaseclib.tvm import TVM
from security.yaseclib.waffles import Waffles


class Features(Flag):
    """
    Available features:
    STARTREK | TVM | IMPULSE | GAP | ABC | MOE | PUNCHER | WAFFLES | STAFF | IDM
    Use "|" operator to require multiple features
    """

    NONE = 0
    STARTREK = auto()
    TVM = auto()
    IMPULSE = auto()
    __GAP = auto()
    GAP = __GAP | TVM
    ABC = auto()
    __MOE = auto()
    MOE = __MOE | TVM
    PUNCHER = auto()
    __WAFFLES = auto()
    WAFFLES = __WAFFLES | TVM
    STAFF = auto()
    IDM = auto()

    def __str__(self):
        members, _ = _decompose(self.__class__, self._value_)
        return ", ".join(
            filter(
                lambda m: "_" not in m,
                map(lambda m: str(m._name_ or m._value_), members),
            )
        )


class AbstractService(object):
    def __init__(self, config):
        super(AbstractService, self).__init__()
        self._config = config

        # Override requirements for your module to use features
        self._require: Features = self.requires()

        # By default everything is disabled
        self._startrek: Optional[Startrek] = None
        self._tvm: Optional[TVM] = None
        self._impulse: Optional[ImpulseQueuer] = None
        self.__calendar_tvm_ticket = None
        self._gap: Optional[Gap] = None
        self._abc: Optional[Abc] = None
        self._staff: Optional[Staff] = None
        self.__moe_tvm_ticket = None
        self._moe: Optional[Moe] = None
        self._puncher: Optional[Puncher] = None
        self.__waffles_tvm_ticket = None
        self._waffles: Optional[Waffles] = None
        self._idm: Optional[Idm] = None

        self.__setup()

    @abstractmethod
    def requires(self) -> Features:
        """
        Implement this method in your child class to specify yaseclib objects you would like to use.
        Return Features object
        """

    def __setup(self):
        """
        Method initialises self context with instances of
        _startrek, _tvm, _impulse, _gap, _abc, _staff, _moe, _waffles
        according to supplied requirements via requires() call
        """
        if Features.STARTREK in self._require:
            self._startrek = Startrek(
                useragent=self._config.get("st", "ua"),
                base_url=self._config.get("st", "url"),
                token=self._config.get("st", "token"),
            )
        if Features.TVM in self._require:
            self._tvm = TVM(
                client_id=self._config.get("tvm", "client_id"),
                client_secret=self._config.get("tvm", "client_secret"),
                destinations=self.__config_getlist("tvm", "destinations"),
            )
        if Features.IMPULSE in self._require:
            self._impulse = ImpulseQueuer(
                base_url=self._config.get("impulse", "url"),
                token=self._config.get("impulse", "token"),
            )
        if Features.GAP in self._require:
            self.__calendar_tvm_ticket = self._tvm.get_service_ticket(
                Gap.CALENDAR_YATEAM_TVM_ID
            )
            self._gap = Gap(
                base_url=self._config.get("gap", "url"),
                token=self._config.get("gap", "token"),
                base_calendar_url=self._config.get("calendar", "url"),
                calendar_tvm_service_ticket=self.__calendar_tvm_ticket,
            )
        if Features.ABC in self._require:
            self._abc = Abc(
                base_url=self._config.get("abc", "url"),
                token=self._config.get("abc", "token"),
            )
        if Features.STAFF in self._require:
            self._staff = Staff(
                base_url=self._config.get("staff", "url"),
                token=self._config.get("staff", "token"),
            )
        if Features.MOE in self._require:
            self.__moe_tvm_ticket = self._tvm.get_service_ticket(Moe.DEFAULT_MOE_TVM_ID)
            self._moe = Moe(tvm_ticket=self.__moe_tvm_ticket)
        if Features.PUNCHER in self._require:
            self._puncher = Puncher(
                base_url=self._config.get("puncher", "url"),
                token=self._config.get("puncher", "token"),
            )
        if Features.WAFFLES in self._require:
            self.__waffles_tvm_ticket = self._tvm.get_service_ticket(Waffles.TVM_ID)
            self._waffles = Waffles(tvm_ticket=self.__waffles_tvm_ticket)
        if Features.IDM in self._require:
            self._idm = Idm(
                base_url=self._config.get("idm", "url"),
                token=self._config.get("idm", "token"),
            )

    def _oauth(self) -> str:
        logging.warning("Accessed OAuth token")
        return self._config.get("st", "token")
