# -*- coding: utf-8 -*-

import configparser
import logging
import re
from functools import wraps
from typing import Callable, Optional, Iterable
from urllib.parse import urlparse
from urllib.parse import urlunparse

import requests

from security.c3po.components.core.common import SIB_PEOPLE, sectask_mapping
from security.c3po.components.core.plugins import get_all_plugin_types
from startrek_client.collections import Issues

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)

CREATOR_RE = re.compile(
    r'\(https://staff.yandex-team.ru/(.*)\)\s*—\s*invited.*security\sto\scomment'
)


def meta_load_plugins(config, filter_f: Callable = lambda p: True):
    return [plugin(config) for plugin in filter(filter_f, get_all_plugin_types())]


def get_config(config_file: str):
    config = configparser.ConfigParser()
    config.read(config_file)
    return config


def safe_func(func: Callable) -> Callable:
    """
    Decorator to make function safe in terms of throwing exceptions.
    """

    @wraps(func)
    def _wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            logger.exception(e)
            return None

    return _wrapper


class OuputManager:
    debug = False

    def __init__(self, debug=False):
        self.debug = debug

    def dbg(self, s):
        if self.debug:
            print(s)


def read_file(filename, charset="utf-8"):
    with open(filename, "r") as f:
        return f.read().decode(charset)


def is_yandex_service(href):
    excludes = ["internet.yandex", "help.yandex.ru/search", "yandex.ru/search/advanced"]
    # Exclude some services
    for e in excludes:
        if e in href:
            return False
    # Yandex service is *yandex*
    if "yandex" in href:
        return True
    return False


def normalize(url):
    o = urlparse(url)
    path = "/" if o.path == "" else o.path
    return urlunparse(("", o.netloc, path, o.params, "", ""))


def get_all_yandex_services():
    services = set()

    try:
        r = requests.get(
            "https://www.yandex.ru/all", verify="/etc/ssl/certs/ca-certificates.crt"
        )
        links = re.findall(r'href=[\'"]?([^\'" >]+)', r.text)
    except Exception as e:
        print(e)
        links = []

    addons = [
        "//wordstat.yandex.ru",
        "//passport.yandex.ru",
        "//yandex.ru/company/",
        "//radio.yandex.ru",
        "//www.yandex.ru",
    ]

    links += addons
    for l in links:
        href = normalize(l)
        if is_yandex_service(href):
            services.add("https:" + href)
    return services


def is_from_sib(login: str):
    return login.lower() in SIB_PEOPLE


@safe_func
def get_duty_tag(issue) -> Optional[str]:
    duty_tags = set(sectask_mapping.keys())
    for tag in issue.tags:
        if tag in duty_tags:
            return tag


EMAIL_BOTS = {
    'zomb-prj-191'  # Yandex Separator
}


def valid_staff_person(login: str) -> bool:
    """
    Method not reliable. Tries to deduce whether certain login is a human without staff API call
    :param logins:weather
    :return:
    """
    return login.lower() not in {
        'st',
        'security'
    }


def __parse_description(issue):
    author = CREATOR_RE.search(issue.description)
    if author:
        return author.group(1)


def filter_non_humans(logins: Iterable[str]):
    """
    Method not reliable. Tries to deduce whether certain login is a human or a robot/zomb without staff API call
    :param logins:weather
    :return:
    """
    return filter(
        lambda login: not login.startswith('robot') and not login.startswith('zomb'),
        logins
    )


def get_iss_tag(issue: Issues) -> Optional[str]:
    tag: str
    for tag in issue.tags:
        if tag.startswith('iss_'):
            return tag
    return


def get_issue_author(issue) -> str:
    """
    Gets author from issue.createdBy.login field by default.
    If author is email bot - parses author from issue.emailFrom field. If does not look like a valid login -
    checks description. If unable to determine author from description - gets first not robot and not zomb from access.
    Otherwise fallbacks to createdBy.login
    :param issue: st Issue instance
    :return: login
    """
    author = issue.createdBy.login
    if author in EMAIL_BOTS:
        email_from = issue.emailFrom
        if email_from:
            try:
                login, domain = email_from.split('@')
                if valid_staff_person(login) and domain == 'yandex-team.ru':
                    return login
                else:
                    access = list(
                        filter_non_humans(map(lambda user: user.login, issue.access))
                    )
                    description_author = __parse_description(issue)
                    if description_author:
                        return description_author
                    if len(access) > 0:
                        return access[0]
            except Exception as e:
                logger.debug(e)
                # pass
    return author
