import datetime
import logging
from datetime import timedelta
from typing import List, Dict, Set, Optional

from jinja2 import Template

from security.c3po.components.core.plugins import AbstractPlugin
from security.c3po.components.core.service import Features
# noinspection PyUnresolvedReferences
from security.c3po.components.core.util import get_config, safe_func  # noqa: F401
from security.c3po.components.plugins.abc_reviewer.configs import ROLE_DESCRIPTIONS, SENSITIVE_ROLES, \
    ABC_IGNORE_LIST, ABC_ID, FALLBACK_ASSIGNEE, COMPONENTS, summon_people

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class AbcReviewer(AbstractPlugin):
    templates_path = "templates"
    QUEUE = 'ACCESS'
    TICKET_DEADLINE = 30  # days

    # noinspection PyUnusedLocal
    @staticmethod
    def check_limits_ok(service: Dict, role: str, privileged_people: int, people: int) -> bool:
        if role == 'tvm_management':
            return privileged_people < 6
        return (
            privileged_people < 7 or
            privileged_people < 0.35 * people
        )

    class Ctx:
        def __init__(self, root_service: int, components: List, tags: List, sec_duty):
            self.root_service = root_service
            self.components = components
            self.tags = tags
            self.sec_duty = sec_duty
            self.services: Dict[int, Dict] = dict()
            self._responsibles: Dict[int, List] = dict()

            self._current_service = None
            # self._redundant_entries = []

        def __mem_field(self, key, value):
            service_id = self._current_service['id']
            d = self.services.get(service_id, dict())
            d.update({key: value})
            self.services.update({service_id: d})
            return value

        def __get_field(self, field, default=None):
            service_id = self._current_service['id']
            return self.services[service_id].get(field, default)

        def current_service(self, service: Dict) -> Dict:
            self._current_service = service
            return self.__mem_field('info', service)

        def people(self, people: List) -> List:
            return self.__mem_field('people', people)

        def service_info(self, info: Dict) -> Dict:
            return self.__mem_field('info', info)

        def assignee(self, assignee: str) -> str:
            return self.__mem_field('assignee', assignee)

        def responsibles(self, responsibles: List) -> List:
            return self.__mem_field('responsibles', responsibles)

        def count_people(self) -> int:
            service_id = self._current_service['id']
            return len(self.services[service_id]['people'])

        @safe_func
        def memorise_redundant_privileges(
            self,
            service: Dict,
            privileged_people: List,
            role_scope: str,
            role_codes: List[str]
        ):
            role = ROLE_DESCRIPTIONS.get(
                role_scope, dict(name=role_scope, description=', '.join(role_codes))
            )
            role_name = role['name']
            role_description = role['description']
            service_slug = service['slug']
            n = len(privileged_people)
            self.__mem_field('entries', self.__get_field('entries', []) + [dict(
                privileged_people=privileged_people,
                role=role_name,
                scope_link=f'https://abc.yandex-team.ru/services/{service_slug}?scope={role_scope}',
                n=n,
                percent=round(n * 100 / self.count_people(), 2),
                description=role_description
            )])

        @safe_func
        def gen_ticket_meta(self, template: Template, service_id: int) -> Optional[Dict]:
            data = self.services[service_id]
            if 'entries' not in data:
                return
            service = data['info']
            service_slug = service['slug']
            entries = data['entries']
            followers = data['responsibles']
            assignee = data['assignee']
            description = template.render(
                link=f'https://abc.yandex-team.ru/services/{service_slug}',
                name=service_slug,
                redundant_entries=entries
            )
            today = datetime.date.today()
            deadline = today + timedelta(days=AbcReviewer.TICKET_DEADLINE)
            return dict(
                summary=f'Аудит доступов ABC: {service_slug}',
                description=description,
                deadline=deadline.isoformat(),
                components=self.components,
                tags=self.tags,
                followers=followers,
                assignee=assignee,
                security='Yes',
                abcService=service_id
            )

        def gen_tickets_meta(self, template: Template) -> List[Dict]:
            tickets = []
            for service_id in self.services.keys():
                ticket_meta = self.gen_ticket_meta(template, service_id)
                if ticket_meta:
                    tickets.append(ticket_meta)

            return tickets

        def __str__(self) -> str:
            service_slug = self._current_service['slug']
            service_id = self._current_service['id']
            return f'{service_slug} (id={service_id})'

    def requires(self) -> Features:
        return Features.STARTREK | Features.STAFF | Features.IDM | Features.ABC

    def main(self):
        shifts = self._abc.get_shifts(ABC_ID.SECURITY)
        duty_mapping = dict()
        for shift in shifts:
            shift_slug = shift["schedule"]["slug"]
            duty_assignee = shift["person"]["login"]
            duty_mapping.update({
                shift_slug: duty_assignee
            })

        self._process_services_and_descendants(AbcReviewer.Ctx(
            root_service=ABC_ID.MARKET,
            components=[COMPONENTS.MARKET],
            tags=['abc_review'],
            sec_duty=duty_mapping.get('market_everyday', FALLBACK_ASSIGNEE)
        ))
        # Add more here:
        # self.process_services_and_descendants(...)
        # ...

    def already_created(self) -> Set[int]:
        already_exists_query = """Queue: ACCESS Status: !Closed Tags: abc_review"""
        issues = self._startrek.issues.find(already_exists_query)
        return set(map(lambda i: int(i.abcService[0].id), filter(lambda i: i.abcService, issues)))

    @safe_func
    def _process_services_and_descendants(self, ctx: Ctx):
        abc_id = ctx.root_service
        services = self._abc.get_service_by_id_with_descendants(abc_id)
        template = self._env.get_template('abc_review_privileges.tpl')
        already_created = self.already_created()
        for service in services:
            ctx.current_service(service)
            service_id = service['id']
            if service_id in already_created or service_id in ABC_IGNORE_LIST:
                logger.debug(f'Skipping id={service_id}')
                continue
            logger.info(f'Auditing service {str(ctx)}')
            ctx.people(
                self._abc.get_people_by_id(service_id, with_descendants=False, use_inheritance_settings=True)
            )
            for role_scope, role_code in SENSITIVE_ROLES.items():
                self._process_service_roles(ctx, service, role_scope, role_code)

            ticket_meta = ctx.gen_ticket_meta(template, service_id)
            if ticket_meta:
                issue = self._startrek.issues.create(
                    queue=AbcReviewer.QUEUE,
                    **ticket_meta
                )
                if issue and summon_people:
                    issue.comments.create(
                        summonees=issue.assignee
                    )

    @safe_func
    def _process_service_roles(
        self,
        ctx: Ctx,
        service: Dict,
        role_scope: str,
        role_codes: List,
    ):
        service_id = service['id']
        privileged_people = self._abc.get_people_by_id(
            service_id, with_descendants=False, use_inheritance_settings=True,
            role_scope=role_scope, role_codes=role_codes
        )

        if not AbcReviewer.check_limits_ok(
            service, role_scope, len(privileged_people), ctx.count_people()
        ):
            logger.warning(f'Too many people with role {role_scope} in {str(ctx)}')
            ctx.responsibles(
                self._abc.get_service_responsibles(service_id)
            )
            assignee = self._abc.get_service_head_by_id(service_id)
            if not assignee:
                assignee = ctx.sec_duty
            ctx.assignee(assignee)
            ctx.memorise_redundant_privileges(service, privileged_people, role_scope, role_codes)
        else:
            logger.info(f'Service {str(ctx)} role scope {role_scope} PASSED check')


# Uncomment for local tests
# def main(config_file: str):
#     config = get_config(config_file)
#     abc_reviewer = AbcReviewer(config)
#     abc_reviewer.main()
#
#
# if __name__ == '__main__':
#     main(
#         'generic_config_testing.ini',
#     )
