import json
import logging
from typing import Iterable

import requests
from security.c3po.components.core.plugins import AbstractPlugin, Features
from security.c3po.components.core.util import is_from_sib
from startrek_client.collections import Components

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class BlockerConcealer(AbstractPlugin):
    title = "Blocker Concealer"
    description = (
        "Searches startrek for ticket with `Security: Yes` and `Priority: Blocker` and "
        "sets (or creates) `Secret` component"
    )
    __checked_queues_file = "checked_queues.json"
    __NOT_SECRET = "not_secret"
    __SECRET = "Secret"

    @staticmethod
    def __get_component(components: Iterable, name: str):
        for component in components:
            if component.name == name:
                return component

    @staticmethod
    def __get_tag(tags: Iterable, name: str):
        for tag in tags:
            if tag == name:
                return tag

    def setup(self):
        filepath = self.resource_path() / BlockerConcealer.__checked_queues_file
        if filepath.exists():
            with open(str(filepath)) as f:
                self._checked_queues = set(json.loads(f.read()))
        else:
            self._checked_queues = set()
        self._headers = {
            "Content-Type": "application/json",
            "Authorization": f"OAuth {self._oauth()}",
        }
        secret_access = {
            "read": {"roles": ["access"]},
            "write": {
                "roles": ["assignee", "follower", "author", "queue-lead"],
                "groups": [54],
                "users": [],
            },
            "grant": {"groups": [8652]},
            "create": {"groups": [962]},
        }
        self._sec_policy = json.dumps(secret_access)

    def main(self):
        st_query = (
            "Security: notEmpty() "
            "Resolution: empty() "
            "(Priority: blocker OR "
            '"Security Severity": "5" OR '
            '"Security Severity": "6") '
            "Components: !Secret "
            "Queue: !IS "
            "Queue: !FINTECHADMIN"
        )
        issues = self._startrek.issues.find(st_query)
        for issue in issues:
            self._conceal_blocker(issue)

    def requires(self) -> Features:
        return Features.STARTREK

    def __check(self, secret_component: Components):
        queue = secret_component.queue.key
        if queue in self._checked_queues:
            return
        self.__reset_secret_component(secret_component)
        self._checked_queues.add(queue)

    def __reset_secret_component(self, component):
        logger.info(f"Resetting Secret component for queue {component.queue.key}")
        sec_comp_id = component.id
        url = f"http://st-api.yandex-team.ru/v2/components/{sec_comp_id}/permissions"
        r = requests.get(url, headers=self._headers)
        version = r.json()["version"]
        url += f"?version={version}"
        requests.patch(url, data=self._sec_policy, headers=self._headers)

    def _conceal_blocker(self, issue):
        secret_component = BlockerConcealer.__get_component(issue.queue.components, BlockerConcealer.__SECRET)
        if not secret_component:
            secret_component = self._startrek.components.create(
                name=BlockerConcealer.__SECRET, queue=issue.queue.key
            )
            sec_comp_id = secret_component.id
            url = (
                f"http://st-api.yandex-team.ru/v2/components/{sec_comp_id}/permissions"
            )
            r = requests.get(url, headers=self._headers)
            version = r.json()["version"]
            url += f"?version={version}"
            self._checked_queues.add(issue.queue.key)
            requests.patch(url, data=self._sec_policy, headers=self._headers)
        else:
            self.__check(secret_component)
        if BlockerConcealer.__get_tag(issue.tags, BlockerConcealer.__NOT_SECRET):
            logger.debug(
                f"{issue.key}: "
                f"Skipped setting {BlockerConcealer.__SECRET} component due to"
                f" {BlockerConcealer.__NOT_SECRET}"
            )
        else:
            issue.components.append(secret_component.id)
            issue.update(components=issue.components, ignore_version_change=True)
            logger.info(f"{issue.key}: Set Secret component")

    def add_secret(self, issue: str):
        if type(issue) == str:
            issue = self._startrek.issues[issue]
        self._conceal_blocker(issue)

    def check_changelog(self, issue: str):
        if type(issue) == str:
            issue = self._startrek.issues[issue]
        changelog = reversed(list(issue.changelog))  # changelog is ordered by date and time

        # we need to check that last change was made by authorised person
        def change_contains(change, component_id: str):
            if change is None:
                return False
            for ch in change:
                if str(ch.name) == component_id:
                    return True
            return False

        for change in changelog:
            login = change.updatedBy.id
            for field in change.fields:
                if field.get('field').id == 'components':
                    ch_from, ch_to = field.get('from'), field.get('to')
                    if change_contains(ch_to, BlockerConcealer.__SECRET):
                        return
                    if change_contains(ch_from, BlockerConcealer.__SECRET) and \
                            not change_contains(ch_to, BlockerConcealer.__SECRET):
                        if not is_from_sib(login):
                            logger.warning(
                                f'{issue.key}: {login} REMOVED {BlockerConcealer.__SECRET} at {change.updatedAt}'
                            )
                            self._conceal_blocker(issue)
                        return

    def teardown(self):
        filepath = self.resource_path() / BlockerConcealer.__checked_queues_file
        with open(filepath, "w") as f:
            f.write(json.dumps(list(self._checked_queues)))
