# -*- coding: utf-8 -*-
import datetime
import logging
import traceback

from security.c3po.components.core.plugins import BasePlugin
from jinja2 import Environment, FileSystemLoader
from startrek_client import Startrek
from security.yaseclib.abc import Abc


def continue_on_fail(f):
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except:
            logging.exception("[BugbountyPlugin]\n%s\n" % traceback.format_exc())

    return wrapper


class Bugbounty(BasePlugin):
    title = "BugBounty"
    desc = (
        "Назначаем исполнителя на тикеты BUGBOUNTY (сейчас это всегда duty_core_team)"
    )

    def setup(self):
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )
        self.no_remind_tag = self.config.get("plugins.bugbounty", "no_remind")
        # Forming shifts
        # TODO: parse domain field and add service team duties
        self.security_abc_id = self.config.get("plugins.bugbounty", "security_abc_id")
        shifts = self.abc.get_shifts(self.security_abc_id)
        self.core_assignee = None

        for shift in shifts:
            shifts_slug = shift["schedule"]["slug"]
            duty_assignee = shift["person"]["login"]

            if shifts_slug == "coresec_everyday":
                self.core_assignee = duty_assignee
                break

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "templates"
            )
        )

    @continue_on_fail
    def _process_new_bountyreport(self):
        st_query = (
            "Queue: BOUNTYREPORT "
            "Resolution: empty() "
            "Assignee: empty() "
            "Created: < now()"
        )
        issues = self.startrek.issues.find(st_query)
        if self.core_assignee:
            for issue in issues:
                issue.update(assignee=self.core_assignee, ignore_version_change=True)

    @continue_on_fail
    def _summon_after_sla_failed(self):
        template = "summon_sla_bugbounty.tpl"

        st_query = (
            "Queue: BOUNTYREPORT "
            "Status: !closed "
            "Assignee: notEmpty() "
            "Tags: !sla_failed "
            "Filter: 72092"
        )
        issues = self.startrek.issues.find(st_query)
        for issue in issues:
            comment = self.env.get_template(template)
            issue.comments.create(text=comment.render(), summonees=[issue.assignee])
            issue.tags.append("sla_failed")
            issue.update(tags=issue.tags, ignore_version_change=True)

    @continue_on_fail
    def _summon_in_old_issues(self):
        template = "summon_old_bugbounty.tpl"

        st_query = (
            "Queue: BOUNTYREPORT "
            "Status: !closed "
            'Status: !"Need info" '
            "Assignee: notEmpty() "
            '"Sort By": Key ASC'
        )
        issues = self.startrek.issues.find(st_query)
        for issue in issues:
            if self.no_remind_tag in issue.tags:
                self._no_remind(issue)
            elif self._check_update_date(issue) > 7:
                comment = self.env.get_template(template)
                issue.comments.create(text=comment.render(), summonees=[issue.assignee])

    def _no_remind(self, issue):
        if self._check_update_date(issue) > 45:
            new_tags = list(issue.tags)
            new_tags.remove(self.no_remind_tag)
            issue.update(tags=new_tags, ignore_version_change=True)

    def _check_update_date(self, issue):
        today = datetime.datetime.now().date()
        update_date = datetime.datetime.strptime(
            issue.updatedAt[:10], "%Y-%m-%d"
        ).date()
        return (today - update_date).days

    def main(self):
        self._process_new_bountyreport()
        self._summon_after_sla_failed()
        self._summon_in_old_issues()
