# -*- coding: utf-8 -*-
import json
import logging
from datetime import date

import requests

from security.c3po.components.core.plugins import BasePlugin
from security.c3po.components.core.util import get_all_yandex_services

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class Csp(BasePlugin):
    title = "Global CSP status monitoring"
    desc = "Monitoring of current status global CSP project"

    def setup(self):
        self.csp_tester_api_url = self.config.get("plugins.csp", "api_url")
        self.stat_robot = self.config.get("stat", "robot_login")
        self.stat_passwd = self.config.get("stat", "robot_password")
        self.stat_api_url = self.config.get("stat", "url")

    def main(self):
        services = get_all_yandex_services()
        result = []

        for s in services:
            tmp = {"url": s, "enabled": False}
            try:
                r = requests.post(
                    self.csp_tester_api_url,
                    data={"policy": s},
                    verify="/etc/ssl/certs/ca-certificates.crt",
                )
                status = r.json()
                if status["protects"]:
                    tmp["enabled"] = True
            except Exception as e:
                logger.error(f"url={s} error={e}")
            result.append(tmp)

        with_csp = len(list(filter(lambda x: x["enabled"] is True, result)))
        without_csp = len(result) - with_csp

        today = date.today()
        data = [
            (
                {
                    "fielddate": today.isoformat(),
                    "enabled": with_csp,
                    "not_enabled": without_csp,
                }
            )
        ]

        r = requests.post(
            self.stat_api_url,
            headers={
                "StatRobotUser": self.stat_robot,
                "StatRobotPassword": self.stat_passwd,
            },
            data={
                "name": "ExtData/Security/csp",
                "scale": "d",
                "data": json.dumps({"values": data}),
            },
        )
