import json
import os
import re

from jinja2 import Environment, FileSystemLoader

from security.c3po.components.core.plugins import BasePlugin
from startrek_client import Startrek


class DpiaParser(BasePlugin):
    title = u"DPIA_Parser"
    desc = u"Обработка тикетов из очереди DPIA. Тикеты заводятся через https://forms.yandex-team.ru/admin/6765"

    def setup(self):
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "comment_templates"
            )
        )

        self.answers = {}
        json_path = os.path.join(self.resource_path(), 'jsons')
        with open(os.path.join(json_path, "answers.json")) as answers_file:
            self.answers = json.loads(answers_file.read())

    def main(self):
        self._walk_dpia()

    def _walk_dpia(self):
        st_query = "Queue: DPIA " "Resolution: empty() " "Tags: !processed"

        issues = self.startrek.issues.find(st_query)

        for issue in issues:
            self._process_dpia(issue)

    def _process_dpia(self, issue):

        issue.tags.append("processed")

        questions = [
            x.strip("*").strip()
            for x in re.findall(r"\*\*[\w\W]*?\*\*", issue.description)
        ]
        answers = [
            x.strip("%").strip().split("\n")
            for x in re.findall(r"%%[\w\W]*?%%", issue.description)
        ]

        template = "low.tpl"
        for i, question in enumerate(questions):
            if question in self.answers:
                for answer in answers[i]:
                    if answer in self.answers[question]["high"]:
                        template = "high.tpl"
                        break
                    elif answer in self.answers[question]["medium"]:
                        template = "medium.tpl"
        summonee = []
        t = self.env.get_template(template)
        if template != "low.tpl":
            summonee = self.config.get("plugins.dpia_parser", "default_assignee")
        issue.comments.create(text=t.render(), summonees=summonee)

        return issue.update(tags=issue.tags, ignore_version_change=True)
