import datetime
import random

from security.c3po.components.core.plugins import BasePlugin
from jinja2 import Environment, FileSystemLoader
from startrek_client import Startrek
from security.yaseclib.gap import Gap
from security.yaseclib.tvm import TVM


class GrantsReviewReviver(BasePlugin):
    title = u"GrantsReviewReviver"
    desc = u"Обработка тикетов из очереди PASSPORTGRANTS. Тикеты заводятся через https://forms.yandex-team.ru/admin/4901"

    def setup(self):
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(
            Gap.CALENDAR_YATEAM_TVM_ID
        )

        self.gap = Gap(
            base_url=self.config.get("gap", "url"),
            token=self.config.get("gap", "token"),
            base_calendar_url=self.config.get("calendar", "url"),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket,
        )
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "comment_templates"
            )
        )

        self.officers = self._config_getlist(
            "plugins.grants_review_reviver", "officers"
        )
        self.grants_review_code = self.config.getint(
            "plugins.grants_review_reviver", "component"
        )

    def main(self):
        if self.gap.is_workday():
            self._walk_passportgrants()
        self.tvm.stop()

    def _walk_passportgrants(self):
        st_query = (
            "Queue: PASSPORTGRANTS "
            "Resolution: empty() "
            "Tags: !passport "
            "Components: %s "
        )
        st_query = st_query % self.grants_review_code

        issues = self.startrek.issues.find(st_query)

        for issue in issues:
            if self._check_update_date(issue) >= 3:
                self._revive(issue)

    def _revive(self, issue):
        summonee = self._get_summonee(issue)
        if summonee:
            t = self.env.get_template("simple_summon.tpl")
            text = t.render()
            issue.comments.create(text=text, summonees=summonee)

    def _check_update_date(self, issue):
        today = datetime.datetime.now().date()
        update_date = datetime.datetime.strptime(
            issue.updatedAt[:10], "%Y-%m-%d"
        ).date()
        return (today - update_date).days

    def _get_summonee(self, issue):
        if issue.assignee:
            return [issue.assignee.login]
        followers = [
            flw.login
            for flw in issue.followers
            if not (flw.login.startswith("zomb-") or flw.login.startswith("robot-"))
        ]
        followers = list(set(followers) - set(self.officers))
        if followers:
            return [random.choice(followers)]
        return []
