import logging
import re
from typing import Tuple, Dict, Optional, Set

from security.c3po.components.core.common import ABC_ID
from security.c3po.components.core.plugins import AbstractPlugin
from security.c3po.components.core.service import Features
from security.c3po.components.core.util import get_config, safe_func
from security.yaseclib.common.cache import cached
from security.yaseclib.common.utils import DictVal
from security.yaseclib.idm2 import RoleId

# logging.basicConfig()

logger = logging.getLogger()
logger.setLevel(logging.DEBUG)


class IdmChecker(AbstractPlugin):
    _UNIQUE_ID_PATTERN = re.compile(r'service_(\d+)_role_(\d+)')

    def requires(self) -> Features:
        return Features.IDM | Features.ABC

    def setup(self):
        market_services = self._abc.get_service_by_id_with_descendants(ABC_ID.MARKET)
        self._allowed_abc_ids = set(
            map(lambda srv: srv['id'], market_services)  # Process only market children
        ) - {ABC_ID.MARKET}

    def main(self):
        self._idm.build_path(self._abc, 37518)
        rules = self._idm.get_rules_awaiting_approve(
            approver='robot-c3po'
        )
        for rule in rules:
            self._process(rule)

    @staticmethod
    def __parse_value_path(value_path: str):
        *service_slugs, _, _, = value_path.strip('/').split('/')
        return service_slugs

    def __parse_unique_id(self, unique_id) -> Tuple[int, int]:
        m = IdmChecker._UNIQUE_ID_PATTERN.search(unique_id)
        abc_id, role_id = m.groups()
        return int(abc_id), int(role_id)

    def _check_employee_in(self, login: str, abc_id: int) -> bool:
        employees = self._abc.get_people_by_id(abc_id)
        return login in employees

    @cached
    def _get_member_info(self, abc_id: int) -> Tuple[Dict[str, Set[int]], Dict[int, Set[str]]]:
        members_info = DictVal(self._abc.get_members_info_by_id(abc_id, fields='person,role'))
        role_login_mapping = dict()
        login_role_mapping = dict()
        for entry in members_info:
            login = entry.person.login
            role_id = entry.role.id
            role_scope = login_role_mapping.get(login, set())
            role_scope.add(role_id)
            login_role_mapping.update({login: role_scope})
            logins = role_login_mapping.get(role_id, set())
            logins.add(login)
            role_login_mapping.update({role_id: logins})

        return login_role_mapping, role_login_mapping

    def _analyse_personal_abc(
        self,
        requested_for: str,
        abc_id: int,
        role_id: int,
        ttl: int
    ) -> Tuple[Optional[bool], str]:
        automation_scope = {RoleId.TVM_MANAGER, RoleId.TVM_SSH_USER}
        if role_id not in automation_scope:
            return None, 'Role out of scope of automation'

        employee_to_roles, roles_to_employees = self._get_member_info(abc_id)

        if requested_for not in employee_to_roles:
            # User has no roles in ABC
            # Falling back to manual approve
            return None, f'User is not in abc id={abc_id}'

        roles = employee_to_roles[requested_for]

        heads_counter = len(roles_to_employees.get(RoleId.HEAD_OF_PRODUCT, []))
        if requested_for in roles_to_employees.get(RoleId.HEAD_OF_PRODUCT, set()):  # If employee is a service head
            if heads_counter > 5:
                logins = ','.join(roles_to_employees[RoleId.HEAD_OF_PRODUCT])
                logging.warning(f'Heads: {logins}')
                return None, f'Too many heads ({heads_counter}) abc_id = {abc_id} - manual approve required'

            if RoleId.HEAD_OF_PRODUCT in roles:
                return True, 'User is a head of product'

        if role_id == RoleId.TVM_SSH_USER:
            if RoleId.DEVELOPER in roles or RoleId.FUNCTIONAL_TESTER in roles:
                return True, 'User has a developer|tester role in abc'
            if RoleId.TVM_MANAGER in roles:
                return True, 'User has a TVM manager role in abc'
        elif role_id == RoleId.TVM_MANAGER:
            tvm_manager_counter = len(roles_to_employees.get(RoleId.TVM_MANAGER, []))
            if tvm_manager_counter > 10:
                return None, f'Too many TVM managers: {tvm_manager_counter} abc = {abc_id} - manual approve required'
            if tvm_manager_counter == 0:
                return True, f'First TVM manager in abc = {abc_id}'

        return None, 'No idea'

    def _analyse_group_abc(
        self,
        group_type: str,
        group_slug: str,
        abc_id: int,
        role_id: int,
        ttl: int
    ) -> Tuple[Optional[bool], str]:
        automation_scope = {RoleId.TVM_MANAGER, RoleId.TVM_SSH_USER}
        if role_id not in automation_scope:
            return None, 'Role out of scope of automation'

        if role_id == RoleId.TVM_MANAGER:
            return False, 'TVM manager forbidden for groups'

        return None, 'No idea'

    @safe_func
    def _process(self, rule):
        if rule.approved is not None:
            return

        if rule.role.node.system.slug != 'abc':
            logging.warning('Processing of non abc rules is not allowed')
            return

        requester = rule.requester
        if requester.type != 'user':
            raise NotImplementedError('Not supported')
        requester_username = requester.username
        ttl = rule.role.ttl_days
        abc_id, role_id = self.__parse_unique_id(rule.role.node.unique_id)
        if abc_id not in self._allowed_abc_ids:
            logging.warning(f'Processing abc_id = {abc_id} is not allowed')
            return

        user = rule.role.user
        group = rule.role.group
        if user:
            requested_for = rule.role.user.username
            logger.debug('\n'.join((
                f'requester: {requester_username}',
                f'requested_for: {requested_for}',
                f'abc_id: {abc_id}',
                f'role_id: {role_id}',
                f'TTL: {ttl}',
            )))
            verdict, comment = self._analyse_personal_abc(requested_for, abc_id, role_id, ttl)
            if verdict is not None:
                logger.info(f'Rule id={rule.id} comment={comment} verdict={verdict}')
                ask_ok = input('OK? > ')  # TODO: Temporary hook for debug purposes
                if ask_ok.strip().lower() == 'ok':
                    self._idm.resolve_rule(rule.id, verdict, comment)
                else:
                    logger.warning(f'Halt {role_id}')
            else:
                logger.debug(f'Skip {role_id}: {verdict} ({comment})')

        if group:
            group_type = group.type
            if group_type == 'department':
                staff_slug = group.slug
                print(requester_username, staff_slug, abc_id, role_id, ttl, '', sep='\n')
                verdict, comment = self._analyse_group_abc(group_type, staff_slug, abc_id, role_id, ttl)
                if verdict is not None:
                    logger.info(f'Rule id={rule.id} comment={comment} verdict={verdict}')
                    ask_ok = input('OK? > ')  # TODO: Temporary hook for debug purposes
                    if ask_ok.strip().lower() == 'ok':
                        self._idm.resolve_rule(rule.id, verdict, comment)
            else:
                raise NotImplementedError(f'Not supported group type {group_type}')


if __name__ == '__main__':
    config_path = '...'
    config = get_config(config_path)
    idm_checker = IdmChecker(config)
    idm_checker.main()
