# -*- coding: utf-8 -*-
import datetime
import logging
import os.path
import random
import re
from urllib.parse import urlparse

import requests
from jinja2 import Environment, FileSystemLoader

from security.c3po.components.core.plugins import BasePlugin
from startrek_client import Startrek
from security.yaseclib.abc import Abc
from security.yaseclib.crt import Crt
from security.yaseclib.gap import Gap
from security.yaseclib.staff import Staff
from security.yaseclib.tvm import TVM

logger = logging.getLogger(__name__)


# flake8: noqa
# TODO: @ezaitov, please, fix flake issues

class Pki(BasePlugin):
    title = 'pki'
    desc = 'Monitoring of issues in SECTASK/pki'

    PASSPORT_TLDS = (
        'ru',
        'com.tr',
        'kz',
        'ua',
        'by',
        'az',
        'com.am',
        'com.ge',
        'com.ua',
        'co.il',
        'kg',
        'lt',
        'lv',
        'md',
        'tj',
        'tm',
        'uz',
        'fr',
        'ee',
        'fi',
        'pl',
        'eu'
    )

    # TODO: move to config
    VALID_AUTORENEWERS = ['ezaitov', 'a-abakumov', 'buglloc', 'robot-c3po']

    SECAUDIT_URL_RE = re.compile(
        'URL \u043f\u0440\u043e\u0434\u0430: (.*)\n\u0418'
    )

    QLOUD_DOMAIN_RE = re.compile(
        'Домен: \(\((.*)\)\)\n'
    )

    DOMAIN_LIST_RE = re.compile(
        'Для хостов:\n(.*?)\n\n', re.DOTALL
    )

    SINGLE_DOMAIN_RE = re.compile(
        '\*\s([\*A-Za-z0-9\.\-]+)\n?'
    )

    BALANCER_TYPE_RE = re.compile(
        'Тип балансера: (\w+)\n'
    )

    REQUESTER_RE = re.compile(
        r'Запросил\(а\)\: \*\*кто\:(.*) \((.*)\)\*\*\n'
    )

    QLOUD_REQUESTER_RE = re.compile(
        'Заказал: (.*) \((\w+)\)\n'
    )

    QLOUD_ENV_RE = re.compile(
        'Окружение: \(\((.*)\)\)\n'
    )

    REISSUE_QLOUD_RE = re.compile(
        'Перезаказ сертификата в связи с истечением срока годности\n'
    )

    SECAUDIT_TICKET_RE = re.compile(
        r'(SECAUDIT-(\d+))'
    )

    CRT_ID_RE = re.compile('https://crt-api.yandex-team.ru/console/#!/certificate/(\d+)/')

    ABC_ID_RE = re.compile(u'ABC ID: (\d+)')

    REISSUE_AWACS_RE = re.compile('Предыдущий тикет на заказ сертификата: (.*)')
    NEW_AWACS_RE = re.compile('Сертификат заказан через awacs - новый заказ.')

    CRT_ABC_ID = 770
    resolved_statuses = ['closed', 'approved']
    need_follow = False
    ok_summon = True
    ok_summon_assignee = True

    balancer_type_question_text = 'Пожалуйста, используйте InternalCA для внутреннего балансера ' \
                                  'или расскажите, зачем нужен именно ProductionCA.'

    manual_link_secaudit_text = 'Не удалось автоматически обнаружить тикет на SECAUDIT для всех доменов, пожалуйста, ' \
                                'привяжите руками. '

    manual_link_secaudit_text += 'Если тикета еще нет — его можно завести ' \
                                 '((https://wiki.yandex-team.ru/product-security/audit/ здесь)).'

    filter_qloud_junk_text = 'В junk-проектах нельзя использовать ProductionCA.'

    filter_homeworkers_text = 'Внимание, заказывающий является надомником!'

    filter_dismissed_text = 'Внимание, заказывающий уволился! Служебные уведомления могут теряться.'

    closed_service_text = 'Внимание, заказывается сертификат на ((https://abc.yandex-team.ru/service/{}/ закрытый сервис)).'

    filter_not_platform = 'Пожалуйста, расскажите, почему вы заказываете сертификат сервиса на свою ' \
                          'учетную запись, а не используете механизмы автоматического заказа AWACS?' \
 
    summon_on_migration = 'Если вы переезжаете в AWACS, это новый заказ сертификата, у вас есть ' \
                                  'закрытый тикет на SECAUDIT, но мы не ответили вам рамках SLA - пожалуйста, ' \
                                  'явно призовите исполнителя тикета.'

    move_redirect_domain_to_any = 'Сервис на указанном домене отвечает редиректом. ' \
                                  'Домены-редиректоры нужно перенести за ' \
                                  '((https://wiki.yandex-team.ru/l7/any/ ANY)) и удалить из окружения Qloud.'

    too_deep_domain = 'Вы заказываете сертификат для домена 4 уровня с префиксом www или m.' \
                      'Если вам действительно необходим сертификат для этого домена - ' \
                      'пожалуйста, предоставьте статистику запросов на этот домен за предыдущий месяц.' \
                      'Если такой статистики нет или домен не используется - удалите его из окружения Qloud и DNS.'

    got_secaudit = 'Обратите внимание, есть закрытый тикет на SECAUDIT.\n'
    got_migration_ticket = 'Тикет на миграцию в AWACS попадает под SLA.\n'

    fix_role_abc = 'В тикет добавлены все пользователи с ролью "Администрирование" в ABC, ' \
                   'выдайте нужным сотрудникам роль "Ответственный за сертификат".\n' \
                   'Эти люди будут получать уведомления об окончании сроков действия, ' \
                   'а также права на отзыв выданных сертификатов.'

    no_abc_error = 'Выписать сертификат без привязки к ABC больше нельзя.'

    no_secaudit_domains_text = 'В SECAUDIT тикете отсутствуют следующие домены:\n{}\nРазрешенные домены: {}\n'

    no_abc_qloud = 'Укажите ABC ID сервиса в ((https://wiki.yandex-team.ru/' \
                   'qloud/doc/environment/ui/#meta настройках окружения)) ' \
                   'Qloud.'

    reissue_text = 'Если домен не используется — пожалуйста, удалите его из окружения, иначе ' \
                   'сертификат будет перевыпущен автоматически. Помните про грядущее ' \
                   '((https://clubs.at.yandex-team.ru/qe/1227 закрытие Qloud))!'

    autoreissue_text = 'Сертификат будет автоматически перевыпущен {} - **за неделю** до истечения старого. ' \
                       'Если домен больше не используется - удалите его из AWACS, тикет будет закрыт. ' \
                       'Для актуальных доменов ничего делать не нужно, но если новый сертификат не был выписан **за три дня** до истечения старого - ' \
                       'призовите исполнителя.'

    deperson_requesters = ['robot-qloud-client',
                           'robot-market-crt',
                           'robot-awacs-certs',
                           'robot-yc-certmanager']

    CLOUD_DOMAINS = ['yandexcloud.net', 'cloud.yandex.net', 'cloud.yandex.ru', 'cloud.yandex.com',
                     'kms.yandex', 'cr.yandex']

    def setup(self):
        self.is_redirect = False
        self.is_trash = False
        self.need_follow = False
        self.ok_summon = True
        self.ok_summon_assignee = True
        self.abc = Abc(base_url=self.config.get('abc', 'url'), token=self.config.get('abc', 'token'))
        self.tvm = TVM(
            client_id=self.config.get('tvm', 'client_id'),
            client_secret=self.config.get('tvm', 'client_secret'),
            destinations=self._config_getlist('tvm', 'destinations')
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(Gap.CALENDAR_YATEAM_TVM_ID)
        self.gap = Gap(
            base_url=self.config.get('gap', 'url'), token=self.config.get('gap', 'token'),
            base_calendar_url=self.config.get('calendar', 'url'),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket
        )
        self.startrek = Startrek(
            useragent=self.config.get('st', 'ua'), base_url=self.config.get('st', 'url'),
            token=self.config.get('st', 'token')
        )
        self.staff = Staff(self.config.get('staff', 'url'), self.config.get('staff', 'token'))
        self.certor = Crt(self.config.get('plugins.pki', 'url'), self.config.get('plugins.pki', 'token'))
        self.service_mapping = {
            'audap': ['daniilsh'],
            'naumov-al': [
                'aleshin',
                'ivanov',
                'shibanov',
                'golubtsov'
            ],
            'kaleda': ['grishakov'],
            'ybuchnev': [
                'rommich',
                'abash',
                'asavinovsky',
                'aryazanov'
            ]
        }
        self.bosses = {}
        for key, values in self.service_mapping.items():
            for value in values:
                if value in self.bosses:
                    self.bosses[value].append(key)
                else:
                    self.bosses[value] = [key]

    def main(self):
        for issue in self.get_unchecked_issues():
            errors = []
            try:
                self.add_crt_team(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't add CRT team.")

            # Link with SECAUDIT
            try:
                self._link_with_secaudit(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't link to SECAUDIT.")

            try:
                self.add_certificate_service_team(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't add ABC service team.")

            # Check qloud balancer type
            try:
                self._check_qloud_balancer_type(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check balancer type.")

            # Check if junk or junk domain
            try:
                self._check_qloud_junk(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check if qloud junk.")

            # Check if requester is homeworker
            try:
                self._check_homeworker(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check if requester is homeworker.")

            # Check if requester is dismissed
            try:
                self._check_requester_dismissed(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check if requester is dismissed.")

            # Check if requester is not from platform
            try:
                self._check_personal_request(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check if personal request.")

            # Check if abc service is deleted
            try:
                self._check_abc_service_status(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check abc service status.")

            # Auto assign
            try:
                self._auto_assign(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't auto assign the task.")

            # Prepare AWACS tickets for autorenew, summon assignee if required
            try:
                self._check_awacs_reissue(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't prepare autorenew.")

            self._set_checked(issue, errors)

        for issue in self.get_checked_issues():
            # Check qloud env problems for open but processed tickets
            try:
                self._check_qloud_domain(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        for issue in self.get_autorenew_issues():
            # Autorenew pending tickets
            try:
                self._autorenew_pending(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        for issue in self.get_secaudit_not_required_issues():
            # Check if secaudit was provided and contains required domains
            try:
                self._check_secaudit_domains(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        for issue in self.get_invalid_issues():
            # Autoreject cancelled requests
            try:
                self._reject_invalid(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        for issue in self.get_valid_cloudonly_issues():
            # Autoapprove cloud tickets
            try:
                self._autoapprove_cloud_issue(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        for issue in self.get_reissue_marked_issues():
            # Autoreissue later
            try:
                self._schedule_autoreissue(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        self.tvm.stop()

    def _check_qloud_balancer_type(self, issue):
        if issue.createdBy.login != 'robot-crt':
            return
        u, is_qloud, is_awacs = self._get_requester(issue)
        if not is_qloud:
            return
        is_internal = False
        need_reissue = False
        for c in issue.comments.get_all():
            balancer_types = self.BALANCER_TYPE_RE.findall(c.text)
            if not balancer_types:
                continue
            if not balancer_types[0] in ['internal']:
                continue
            is_internal = True
            reissue = self.REISSUE_QLOUD_RE.findall(c.text)
            if reissue:
                need_reissue = True
                break
        if need_reissue:
            issue.comments.create(text=self.reissue_text, summonees=[u])
            return
        if not is_internal:
            return
        self.need_follow = True
        issue.comments.create(text=self.balancer_type_question_text)

    def _autoapprove_cloud_issue(self, issue):
        if issue.createdBy.login != 'robot-crt':
            return
        u, is_qloud, is_awacs = self._get_requester(issue)
        if u != 'robot-yc-certmanager':
            return
        if 'secaudit_missing_domains' in issue.tags:
            return
        if 'secaudit_required' in issue.tags:
            return
        if 'trash' in issue.tags:
            return
        if 'cloud_only_domains' not in issue.tags:
            return
        # check if cloud_only_domains tag was set by our robot
        valid_tag = False
        for log_entry in issue.changelog:
            for l in log_entry.fields:
                if not l.get('field') or l.get('field').id != 'tags':
                    continue
                if log_entry.updatedBy.login not in self.VALID_AUTORENEWERS:
                    continue
                if l.get('to') and 'cloud_only_domains' in l.get('to'):
                    valid_tag = True
                    break
        if not valid_tag:
            return

        cert_ids = self.CRT_ID_RE.findall(issue.description)
        if not cert_ids:
            logging.error('No CRT request ID in cloud ticket, id: {}'.format(issue.key))
            return
        cert = self.certor.get_certificate(cert_ids[0])
        if not cert:
            logging.error('Can not get CRT request, id: {}'.format(cert_ids[0]))
            return
        if cert.get('status') != 'need_approve':
            logging.error('[pki] CRT request status error, id: {}'.format(cert_ids[0]))
            return
        if not cert.get('approve_request'):
            logging.error('[pki] CRT approve request not found, id: {}'.format(cert_ids[0]))
            return
        if not self.certor.approve_request(cert.get('approve_request')):
            logging.error('[pki] CRT approve request error, id: {}'.format(cert.get('approve_request')))

    def _autorenew_pending(self, issue):
        if issue.createdBy.login != 'robot-crt':
            return
        # autorenew is for awacs only
        u, is_qloud, is_awacs = self._get_requester(issue)
        #        if not is_awacs and not is_qloud:
        #            return
        # we approve only 7 days before expiration
        #        print ('autorenew', issue.key)
        if not self.is_deadline_met(issue):
            #            print ('deadline not met', issue.key)
            return
        #        print ('checking tags', issue.key)
        # check if tag was set by security
        valid_tag = False
        if 'reissue_pending' in issue.tags:
            # check if 'reissue_pending' tag was set by security team
            for log_entry in issue.changelog:
                for l in log_entry.fields:
                    if not l.get('field') or l.get('field').id != 'tags':
                        continue
                    if log_entry.updatedBy.login not in self.VALID_AUTORENEWERS:
                        continue
                    if l.get('to') and 'reissue_pending' in l.get('to'):
                        valid_tag = True
                        break
        if not valid_tag:
            #            print ('not valid tag', issue.key)
            return
        cert_ids = self.CRT_ID_RE.findall(issue.description)
        if not cert_ids:
            #            print ('not valid cert id', issue.key)
            return
        cert = self.certor.get_certificate(cert_ids[0])
        if not cert:
            #            print ('cant get cert', issue.key)
            return
        if cert.get('status') != 'need_approve':
            #            print ('invalid status', issue.key)
            return
        if not cert.get('approve_request'):
            return
        if self.certor.approve_request(cert.get('approve_request')):
            ttags = []
            for t in issue.tags:
                if t == 'reissue_pending':
                    continue
                ttags.append(t)
            if is_awacs:
                ttags.append('autorenew')
            issue.update(tags=ttags, ignore_version_change=True)

    #        else:
    #            print ('approve error', issue.key)

    def _reject_invalid(self, issue):
        # XXX: don't let people reject any valid request
        cert_ids = self.CRT_ID_RE.findall(issue.description)
        if not cert_ids:
            return
        cert = self.certor.get_certificate(cert_ids[0])
        if not cert:
            return
        if cert.get('status') != 'need_approve':
            return
        if not cert.get('approve_request'):
            return
        return self.certor.reject_request(cert.get('approve_request'))

    def _schedule_autoreissue(self, issue):
        # XXX: it's almost copypaste of _check_awacs_reissue
        awacs_robot_login = 'robot-awacs-certs'
        if issue.createdBy.login != 'robot-crt':
            return
        # autorenew is for awacs only
        u, is_qloud, is_awacs = self._get_requester(issue)
        if not is_awacs:
            return
        valid_reissue = False
        for log_entry in issue.changelog:
            for l in log_entry.fields:
                if not l.get('field') or l.get('field').id != 'tags':
                    continue
                if log_entry.updatedBy.login not in self.VALID_AUTORENEWERS:
                    continue
                if l.get('to') and 'reissue_ok' in l.get('to'):
                    valid_reissue = True
                    break
        if not valid_reissue:
            return

        need_reissue = False
        valid_old_ticket = False
        deadline = None
        prev_ticket = None
        for c in issue.comments.get_all():
            if c.updatedBy.login == c.createdBy.login and c.createdBy.login == awacs_robot_login:
                reissue = self.REISSUE_AWACS_RE.findall(c.text)
                if reissue and reissue[0] != '-':
                    prev_ticket = self.startrek.issues[reissue[0]]
                    current_abc_id = self.get_issue_abc(issue)
                    if current_abc_id and current_abc_id == self.get_issue_abc(prev_ticket):
                        valid_old_ticket = True
                        break

        if prev_ticket and valid_old_ticket:
            # find old certificate id
            cert_ids = self.CRT_ID_RE.findall(prev_ticket.description)
            if cert_ids:
                # get old certificate details and expiration date
                cert = self.certor.get_certificate(cert_ids[0])
                if cert:
                    prev_cert_end_date = self.certor.load_date(cert.get('end_date'))
                    if prev_cert_end_date:
                        # count new deadline (10 days before expiration)
                        deadline = prev_cert_end_date - datetime.timedelta(10)
                        need_reissue = True
            # let's link with old tickets
            try:
                issue.links.create(issue=prev_ticket.key, relationship='relates')
            except Exception:
                pass
            # let's check that it has SECAUDIT tickets linked
            for linked in prev_ticket.links:
                if not hasattr(linked, 'object'):
                    continue
                if 'SECAUDIT' in linked.object.key:
                    try:
                        issue.links.create(issue=linked.object.key, relationship='relates')
                    except Exception:
                        pass

        if need_reissue and deadline:
            # mark ticket as autorenew
            deadline = datetime.datetime.strftime(deadline, '%Y-%m-%d')
            issue_tags = []
            for tag in issue.tags:
                if tag == 'issue_ok':
                    continue
                issue_tags.append(tag)
            issue_tags.append('reissue_pending')
            issue.update(tags=issue_tags,
                         deadline=deadline,
                         assignee='ezaitov',
                         ignore_version_change=True)
            issue.comments.create(text=self.autoreissue_text.format(deadline))
            transition = issue.transitions['start_progress']
            transition.execute()

    def _check_awacs_reissue(self, issue):
        awacs_robot_login = 'robot-awacs-certs'
        if issue.createdBy.login != 'robot-crt':
            return
        # autorenew is for awacs only
        u, is_qloud, is_awacs = self._get_requester(issue)
        if not is_awacs:
            return
        need_reissue = False
        need_summon_on_migration = False
        valid_old_ticket = False
        deadline = None
        prev_ticket = None
        for c in issue.comments.get_all():
            if c.updatedBy.login == c.createdBy.login and c.createdBy.login == awacs_robot_login:
                reissue = self.REISSUE_AWACS_RE.findall(c.text)
                if reissue and reissue[0] != '-':
                    prev_ticket = self.startrek.issues[reissue[0]]
                    current_abc_id = self.get_issue_abc(issue)
                    if current_abc_id and current_abc_id == self.get_issue_abc(prev_ticket):
                        # ok, we got approved ticket with same abc id
                        if 'autorenew' in prev_ticket.tags:
                            # check if 'autorenew' tag was set by security team
                            # TODO: move to separate func
                            for log_entry in prev_ticket.changelog:
                                for l in log_entry.fields:
                                    if not l.get('field') or l.get('field').id != 'tags':
                                        continue
                                    if log_entry.updatedBy.login not in self.VALID_AUTORENEWERS:
                                        continue
                                    if l.get('to') and 'autorenew' in l.get('to'):
                                        valid_old_ticket = True
                                        break
                    break
                elif self.NEW_AWACS_RE.findall(c.text):
                    need_summon_on_migration = True
                    break

        if need_summon_on_migration:
            if 'awacs_new' not in issue.tags:
                issue.update(tags=issue.tags + ['awacs_new'], ignore_version_change=True)

        if 'awacs' not in issue.tags:
            issue.update(tags=issue.tags + ['awacs'], ignore_version_change=True)

        if prev_ticket and valid_old_ticket:
            # find old certificate id
            cert_ids = self.CRT_ID_RE.findall(prev_ticket.description)
            if cert_ids:
                # get old certificate details and expiration date
                cert = self.certor.get_certificate(cert_ids[0])
                if cert:
                    prev_cert_end_date = self.certor.load_date(cert.get('end_date'))
                    if prev_cert_end_date:
                        # count new deadline (10 days before expiration)
                        deadline = prev_cert_end_date - datetime.timedelta(10)
                        need_reissue = True
            # let's link with old tickets
            try:
                issue.links.create(issue=prev_ticket.key, relationship='relates')
            except Exception:
                pass
            # let's check that it has SECAUDIT tickets linked
            for linked in prev_ticket.links:
                if not hasattr(linked, 'object'):
                    continue
                if 'SECAUDIT' in linked.object.key:
                    try:
                        issue.links.create(issue=linked.object.key, relationship='relates')
                    except Exception:
                        pass
        if need_reissue and deadline:
            # mark ticket as autorenew
            deadline = datetime.datetime.strftime(deadline, '%Y-%m-%d')
            issue.update(tags=issue.tags + ['reissue_pending'],
                         deadline=deadline,
                         assignee='ezaitov',
                         ignore_version_change=True)
            issue.comments.create(text=self.autoreissue_text.format(deadline))
            transition = issue.transitions['start_progress']
            transition.execute()
            return
        if need_summon_on_migration and self.ok_summon_assignee and issue.assignee:
            issue.comments.create(text=self.summon_on_migration)
            issue.comments.create(text=self.got_migration_ticket, summonees=[issue.assignee.login])

    def is_deadline_met(self, issue):
        if issue.deadline:
            today = datetime.datetime.now().date()
            deadline = datetime.datetime.strptime(issue.deadline, "%Y-%m-%d").date()
            if (deadline - today).days < 3:
                return True
        return False

    def _check_qloud_junk(self, issue):
        for c in issue.comments.get_all():
            link = self.QLOUD_DOMAIN_RE.findall(c.text)
            if not link:
                continue
            domain = link[0].split()[-1]
            if 'junk.yandex-team.ru' in domain:
                self.need_follow = True
                issue.comments.create(text=self.filter_qloud_junk_text)
                break
            self._check_if_redirect_domain(issue, domain)
            self._check_if_stupid_domain(issue, domain)

    def _get_requester(self, issue):
        is_qloud = False
        is_awacs = False
        if issue.createdBy.login == "robot-crt":
            logins = self.REQUESTER_RE.findall(issue.description)
            if not logins or not logins[0]:
                return '', is_qloud, is_awacs
            if str(logins[0][-1]) == 'robot-qloud-client':
                is_qloud = True
            if str(logins[0][-1]) == 'robot-awacs-certs':
                is_awacs = True
            if str(logins[0][-1]) not in ['robot-qloud-client',
                                          'robot-awacs-certs']:
                return str(logins[0][-1]), is_qloud, is_awacs
            for c in issue.comments.get_all():
                persons = self.QLOUD_REQUESTER_RE.findall(c.text)
                if not persons:
                    continue
                return str(persons[0][-1]), is_qloud, is_awacs
            return str(logins[0][-1]), is_qloud, is_awacs
        else:
            if issue.createdBy.login == 'robot-qloud-client':
                is_qloud = True
            if issue.createdBy.login == 'robot-awacs-certs':
                is_awacs = True
            return issue.createdBy.login, is_qloud, is_awacs

    def _check_homeworker(self, issue):
        username, is_qloud, is_awacs = self._get_requester(issue)
        if self.staff.is_homeworker(username):
            self.need_follow = True
            issue.comments.create(text=self.filter_homeworkers_text)

    def _check_requester_dismissed(self, issue):
        username, is_qloud, is_awacs = self._get_requester(issue)
        if not username or is_qloud:
            return
        if self.staff.is_dismissed(username):
            self.need_follow = True
            issue.comments.create(text=self.filter_dismissed_text)

    def _check_personal_request(self, issue):
        username, is_qloud, is_awacs = self._get_requester(issue)
        if not username:
            return
        if is_awacs:
            self.need_follow = True
        if is_qloud:
            return
        if username not in self.deperson_requesters:
            self.need_follow = True
            self.ok_summon = False

            # do not ask about AWACS on cloud only domains
            domains = self.DOMAIN_LIST_RE.findall(issue.description)
            if not domains:
                return
            requested_domains = set()
            for domain in self.SINGLE_DOMAIN_RE.findall(domains[0]):
                requested_domains.add(domain.lower())

            cloud_domains = set()
            for domain in requested_domains:
                cloud_domain_found = False
                for test_domain in self.CLOUD_DOMAINS:
                    if test_domain == domain:
                        cloud_domain_found = True
                        break
                    if domain.endswith('.{}'.format(test_domain)):
                        cloud_domain_found = True
                        break
                if cloud_domain_found:
                    cloud_domains.add(domain.lower())
            if cloud_domains == requested_domains:
                return

            issue.comments.create(text=self.filter_not_platform, summonees=[username])

    def _slugify_domain(self, domain):
        if 'yandex' in domain and 'yandex-team' not in domain:
            m = 0
            splnetloc = domain.split('.')
            if splnetloc[0] in ['www', 'm']:
                m = 1
            for tld in self.PASSPORT_TLDS:
                ctld = tld.split('.')
                if len(ctld) > 1 and len(splnetloc) > 2 and splnetloc[-1] == ctld[1] and splnetloc[-2] == ctld[0]:
                    ndom = '.'.join(splnetloc[m:-2])
                    return ndom + '.tld', True
                if splnetloc[-1] == tld:
                    ndom = '.'.join(splnetloc[m:-1])
                    return ndom + '.tld', True
        return domain.lower(), False

    def _check_secaudit_domains(self, issue):
        if 'secaudit_processed' in issue.tags:
            return
        domains = self.DOMAIN_LIST_RE.findall(issue.description)
        if not domains:
            return
        requested_domains = set()
        missing_domains = set()
        allowed_domains = set()
        for domain in self.SINGLE_DOMAIN_RE.findall(domains[0]):
            domain_slug, passport_domain = self._slugify_domain(domain)
            missing_domains.add(domain_slug)
            requested_domains.add(domain.lower())

        cloud_domains = set()
        for domain in requested_domains:
            cloud_domain_found = False
            for test_domain in self.CLOUD_DOMAINS:
                if test_domain == domain:
                    cloud_domain_found = True
                    break
                if domain.endswith('.{}'.format(test_domain)):
                    cloud_domain_found = True
                    break
            if cloud_domain_found:
                cloud_domains.add(domain.lower())
            requested_domain_mask, passport_domain = self._slugify_domain(domain)
            for linked in issue.links:
                if not hasattr(linked, 'object'):
                    continue
                if 'SECAUDIT' not in linked.object.key:
                    continue
                sa_issue = self.startrek.issues[linked.object.key]
                if sa_issue.productionURL:
                    attr_uinfo = urlparse(sa_issue.productionURL)
                    if attr_uinfo and attr_uinfo.netloc:
                        netloc = attr_uinfo.netloc
                        if ':' in netloc:
                            netloc = netloc.split(':')[0]
                        allowed_domain_mask, allowed_passport_domain = self._slugify_domain(netloc)
                        allowed_domains.add(allowed_domain_mask)
                        if requested_domain_mask == allowed_domain_mask and requested_domain_mask in missing_domains:
                            missing_domains.remove(requested_domain_mask)
                else:
                    prod_url = self.SECAUDIT_URL_RE.findall(sa_issue.description)
                if not prod_url:
                    continue
                regx_uinfo = urlparse(prod_url[0])
                if regx_uinfo and regx_uinfo.netloc:
                    netloc = regx_uinfo.netloc
                    if ':' in netloc:
                        netloc = netloc.split(':')[0]
                    allowed_domain_mask, allowed_passport_domain = self._slugify_domain(netloc)
                    allowed_domains.add(allowed_domain_mask)
                    if requested_domain_mask == allowed_domain_mask and requested_domain_mask in missing_domains:
                        missing_domains.remove(requested_domain_mask)
        tags = issue.tags
        if requested_domains == cloud_domains:
            tags = tags + [u'cloud_only_domains']

        if missing_domains:
            domain_list = ''
            allowed_domains_list = ''
            for item in missing_domains:
                domain_list += '* {}\n'.format(item)
            for item in allowed_domains:
                allowed_domains_list += '* {}\n'.format(item)
            issue.comments.create(text=self.no_secaudit_domains_text.format(domain_list, allowed_domains_list))
            tags = list(set(tags + [u'secaudit_missing_domains']))
        issue.update(tags=tags + [u'secaudit_processed'], ignore_version_change=True)

    def _link_with_secaudit(self, issue):
        known = set()
        domains = self.DOMAIN_LIST_RE.findall(issue.description)
        if not domains:
            return
        for domain in self.SINGLE_DOMAIN_RE.findall(domains[0]):
            for sa_issue in self.get_secaudit_issues(domain):
                if sa_issue.key in known:
                    continue
                if not sa_issue.description:
                    continue
                prod_url = self.SECAUDIT_URL_RE.findall(sa_issue.description)
                regx_uinfo = None
                attr_uinfo = None
                if prod_url:
                    regx_uinfo = urlparse(prod_url[0])
                if sa_issue.productionURL:
                    attr_uinfo = urlparse(sa_issue.productionURL)
                if regx_uinfo and regx_uinfo.netloc == domain:
                    try:
                        issue.links.create(issue=sa_issue.key, relationship='relates')
                    except Exception:
                        pass
                    known.add(sa_issue.key)
                    continue
                if attr_uinfo and attr_uinfo.netloc == domain:
                    try:
                        issue.links.create(issue=sa_issue.key, relationship='relates')
                    except Exception:
                        pass
                    known.add(sa_issue.key)
                    continue

        secaudit_found = False
        secaudit_closed = False
        for linked in issue.links:
            if not hasattr(linked, 'object'):
                continue
            if 'SECAUDIT' in linked.object.key:
                secaudit_found = True
                if linked.object.status.key in self.resolved_statuses:
                    secaudit_closed = True
                continue
        for c in issue.comments.get_all():
            tickets = self.SECAUDIT_TICKET_RE.findall(c.text)
            for sa_ticket in tickets:
                sa_issue = self.startrek.issues[sa_ticket[0]]
                if sa_issue.key in known:
                    continue
                try:
                    issue.links.create(issue=sa_issue.key, relationship='relates')
                except Exception:
                    pass
                known.add(sa_issue.key)
                secaudit_found = True
                if sa_issue.status.key in self.resolved_statuses:
                    secaudit_closed = True
        if not secaudit_found and not (self.is_trash or self.is_redirect):
            requester, is_qloud, is_awacs = self._get_requester(issue)
            self.need_follow = True
            if requester not in self.deperson_requesters and self.ok_summon:
                issue.comments.create(text=self.manual_link_secaudit_text, summonees=[requester])
                self.ok_summon = False
            else:
                issue.comments.create(text=self.manual_link_secaudit_text)
            if not u'secaudit_required' in issue.tags:
                issue.update(tags=issue.tags + [u'secaudit_required'], ignore_version_change=True)
        else:
            tags = []
            for tag in issue.tags:
                if tag == u'secaudit_required':
                    continue
                tags.append(tag)
            issue.update(tags=tags, ignore_version_change=True)

    def _auto_assign(self, issue):
        if issue.assignee:
            return True

        default_assignee = self.config.get('plugins.pki', 'default_assignee')
        logins = self._config_getlist('plugins.pki', 'any_assignee')
        boss_list = self.staff.get_person_chief_list(issue.createdBy.login)
        not_cloud = {"styskin", "abash"}
        if not_cloud & set(boss_list) == not_cloud:
            boss_list.remove("abash")
        if set(self.bosses.keys()) & set(boss_list):
            boss = [boss for boss in boss_list if boss in self.bosses].pop()
            followers = []
            for follower in issue.followers:
                followers.append(follower.login)

            followers += [
                officer for officer in self.bosses[boss]
                if self.gap.get_working_today(officer)
            ]
            if followers:
                issue.update(followers=followers, ignore_version_change=True)

        if self.config.getboolean('plugins.pki', 'use_gap'):
            try:
                logins = [login for login in logins if self.gap.get_working_today(login)]
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()

        if not logins:
            logins = [default_assignee]

        r = random.SystemRandom()
        # FIXME
        issue = self.startrek.issues[issue.key]
        return issue.update(assignee=r.choice(logins), ignore_version_change=True)

    def get_unchecked_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: empty() ' \
                   'AND Components: %d ' \
                   'AND Tags:! processed ' \
                   % (self.config.getint('plugins.pki', 'component'))
        issues = self.startrek.issues.find(st_query)
        return issues

    def get_checked_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: empty() ' \
                   'AND Components: %d ' \
                   'AND Tags: processed ' \
                   'AND Tags:! updated ' \
                   'AND Tags:! reissue_pending ' % (self.config.getint('plugins.pki', 'component'))
        issues = self.startrek.issues.find(st_query)
        return issues

    def get_secaudit_not_required_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: empty() ' \
                   'AND Components: %d ' \
                   'AND Tags: processed ' \
                   'AND Tags:! secaudit_required ' \
                   'AND Tags:! trash ' \
                   'AND Tags:! secaudit_processed ' \
                   'AND Tags:! reissue_pending ' % (self.config.getint('plugins.pki', 'component'))
        issues = self.startrek.issues.find(st_query)
        return issues

    def get_valid_cloudonly_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: empty() ' \
                   'AND Components: %d ' \
                   'AND Tags: processed ' \
                   'AND Tags:! secaudit_required ' \
                   'AND Tags:! trash ' \
                   'AND Tags: cloud_only_domains ' \
                   'AND Tags:! secaudit_missing_domains ' \
                   'AND Tags:! reissue_pending ' % (self.config.getint('plugins.pki', 'component'))
        issues = self.startrek.issues.find(st_query)
        return issues

    def get_autorenew_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: empty() ' \
                   'AND Components: %d ' \
                   'AND Tags: reissue_pending ' \
                   '"Sort by": Updated ASC' % (self.config.getint('plugins.pki', 'component'))
        return self.startrek.issues.find(st_query)

    def get_invalid_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: "won\'tFix" ' \
                   'AND Components: %d ' \
                   % (self.config.getint('plugins.pki', 'component'))
        return self.startrek.issues.find(st_query)

    def get_reissue_marked_issues(self):
        st_query = 'Queue: SECTASK ' \
                   'AND Resolution: empty() ' \
                   'AND Components: %d ' \
                   'AND Tags: processed ' \
                   'AND Tags: reissue_ok ' \
                   'AND Tags:! reissue_pending ' % (self.config.getint('plugins.pki', 'component'))
        issues = self.startrek.issues.find(st_query)
        return issues

    def _set_checked(self, issue, errors=[]):
        tags = [u'processed']

        if errors:
            tags.append('errors')
        #            issue.comments.create(text='Processing erros: {}'.format('\n'.join(errors)))

        # FIXME
        issue = self.startrek.issues[issue.key]
        issue.update(tags=issue.tags + tags, ignore_version_change=True)

    def get_secaudit_issues(self, domain):
        prev_audits = []
        st_query = 'Queue: SECAUDIT and Description: "{0}"'.format(
            re.sub(r'[^a-zA-Z0-9\.-]+', '', domain)
        )
        issues = self.startrek.issues.find(st_query)
        for item in issues:
            prev_audits.append(item)

        st_query = 'Queue: SECAUDIT "Production URL": "https://{0}/"'.format(
            re.sub(r'[^a-zA-Z0-9\.-]+', '', domain)
        )
        issues = self.startrek.issues.find(st_query)
        for item in issues:
            prev_audits.append(item)

        st_query = 'Queue: SECTASK and Description: "{0}"'.format(
            re.sub(r'[^a-zA-Z0-9\.-]+', '', domain)
        )
        issues = self.startrek.issues.find(st_query)
        for issue in issues:
            for linked in issue.links.get_all():
                if not hasattr(linked, 'object'):
                    continue
                if 'SECAUDIT' in linked.object.key:
                    prev_audits.append(linked.object)
        return prev_audits

    def _check_abc_service_status(self, issue):
        issue = self.startrek.issues[issue.key]
        current_abc_id = self.get_issue_abc(issue)
        abc_service = self.abc.get_service_by_id(current_abc_id)
        state = abc_service.get('content', {}).get('service', {}).get('state', {}).get('id', 'unknown')
        if state == 'closed':
            requester, is_qloud, is_awacs = self._get_requester(issue)
            if requester not in self.deperson_requesters:
                issue.comments.create(text=self.closed_service_text, summonees=[requester])
            else:
                issue.comments.create(text=self.closed_service_text)
            self._reject_invalid(issue)

    def add_crt_team(self, issue):
        issue = self.startrek.issues[issue.key]
        people = self.abc.get_people_by_id(self.CRT_ABC_ID,
                                           with_descendants=False,
                                           role_scope="development")
        return issue.update(access=issue.access + people,
                            ignore_version_change=True)

    def _check_if_redirect_domain(self, issue, domain):
        target = ''
        issue = self.startrek.issues[issue.key]
        try:
            r = requests.get('https://{}/'.format(domain), allow_redirects=False, verify=False)
        except Exception:
            return
        else:
            if r.status_code in [301, 302, 307]:
                target = r.headers.get('Location')
        if not target or 'passport.yandex' in target or 'oauth.yandex' in target:
            return
        requester, is_qloud, is_awacs = self._get_requester(issue)
        if requester not in self.deperson_requesters and self.ok_summon:
            issue.comments.create(text=self.move_redirect_domain_to_any, summonees=[requester])
            self.ok_summon = False
        else:
            issue.comments.create(text=self.move_redirect_domain_to_any)
        self.is_redirect = True

    def _check_if_stupid_domain(self, issue, domain):
        d = domain.split('.')
        if len(d) < 4:
            return
        if d[0] not in ['www', 'm', 'pda', 't']:
            return
        issue = self.startrek.issues[issue.key]
        requester, is_qloud, is_awacs = self._get_requester(issue)
        if requester not in self.deperson_requesters and self.ok_summon:
            issue.comments.create(text=self.too_deep_domain, summonees=[requester])
            self.ok_summon = False
        else:
            issue.comments.create(text=self.too_deep_domain)
        self.is_trash = True

    def _check_qloud_domain(self, issue):
        if u'updated' in issue.tags:
            return
        domain = ''
        requester, is_qloud, is_awacs = self._get_requester(issue)
        if not is_qloud:
            return
        envid = ''
        issue = self.startrek.issues[issue.key]
        for c in issue.comments.get_all():
            link = self.QLOUD_DOMAIN_RE.findall(c.text)
            if not link:
                continue
            domain = link[0].split()[-1]
            envids = self.QLOUD_ENV_RE.findall(c.text)
            if envids and envids[0]:
                envid = envids[0].strip().split()[-1]
            if domain and envid:
                break
        if not domain:
            return
        tags = [u'updated', u'qloud']
        if not envid:
            issue.comments.create(text='Окружение не найдено.', summonees=[issue.assignee.login, requester])
            issue.update(tags=issue.tags + tags, ignore_version_change=True)
        env_data = self.get_qloud_env(envid)
        if not env_data.get('domains'):
            issue.comments.create(text='Домен не найден в окружении "{}".'.format(envid),
                                  summonees=[issue.assignee.login, requester])
            issue.update(tags=issue.tags + tags, ignore_version_change=True)
            return
        domain_found = False
        qloud_check_problems = ''
        summonees = [issue.assignee.login]
        for d in env_data.get('domains'):
            if d.get("domainName") != domain:
                continue
            domain_found = True
            if d.get('issueCertificateTicket') and d.get('issueCertificateTicket') != issue.key:
                qloud_check_problems += '* Сертификат для домена заказан в другом тикете.\n'
                summonees = [requester]
            if not d.get("published"):
                qloud_check_problems += '* Вы заказываете сертификат для неопубликованного домена.\n'
                summonees = [requester]
            if d.get("warning") and 'IPV6 is not' not in d.get('warning', ''):
                qloud_check_problems += '* С доменом есть проблемы:\n {}\n'.format(d.get('warning', ''))
                qloud_check_problems += '\nУдалите домен из окружения, чтобы он перестал перезапрашиваться.\n'
        if not domain_found:
            issue.comments.create(text='Домен не найден в окружении.')
            issue.update(tags=issue.tags + tags, ignore_version_change=True)
            self._reject_invalid(issue)

        if qloud_check_problems:
            abc_id = self.get_issue_abc(issue)
            certificate_admins = self.abc.get_people_by_id(abc_id, with_descendants=False, role_scope="cert")
            for l in certificate_admins:
                summonees.append(l)
            issue.comments.create(text=qloud_check_problems, summonees=summonees)
            issue.update(tags=issue.tags + tags, ignore_version_change=True)

    def get_qloud_env(self, envid):
        try:
            base_url = self.config.get('qloud', 'url')
            token = self.config.get('qloud', 'token')
            r = requests.get(
                '{}/api/v1/environment/stable/{}'.format(base_url, envid),
                headers={'Authorization': 'OAuth {}'.format(token)}
            )
            env_data = r.json()
        except Exception:
            return {}
        return env_data

    def get_issue_abc(self, issue):
        issue = self.startrek.issues[issue.key]
        abcs = self.ABC_ID_RE.findall(issue.description)
        if not abcs:
            return 0
        return int(abcs[0].strip())

    def add_certificate_service_team(self, issue):
        issue = self.startrek.issues[issue.key]
        abc_id = self.get_issue_abc(issue)
        if not abc_id:
            requester, is_qloud, is_awacs = self._get_requester(issue)
            if requester not in self.deperson_requesters and self.ok_summon:
                error_text = self.no_abc_error
                if issue.createdBy.login == 'robot-qloud-client':
                    error_text += '\n{}'.format(self.no_abc_qloud)
                issue.comments.create(text=error_text,
                                      summonees=[requester])
                self._reject_invalid(issue)
                self.ok_summon = False
            else:
                issue.comments.create(text=self.no_abc_error)
            return

        certificate_admins = self.abc.get_people_by_id(abc_id,
                                                       with_descendants=False,
                                                       role_scope="cert")
        if not certificate_admins:
            certificate_admins = self.abc.get_people_by_id(abc_id,
                                                           with_descendants=False,
                                                           role_scope="administration")
            # do not spam too many people
            if not certificate_admins:
                certificate_admins = self.abc.get_people_by_id(
                    abc_id,
                    with_descendants=True,
                    role_scope="administration"
                )
            issue.update(comment=self.fix_role_abc, ignore_version_change=True)
            self.need_follow = True

        if len(certificate_admins) > 20:
            certificate_admins = certificate_admins[:20]

        if self.need_follow:
            followers = []
            for follower in issue.followers:
                followers.append(follower.login)
            issue = self.startrek.issues[issue.key]
            issue.update(followers=followers + certificate_admins,
                         ignore_version_change=True)
        else:
            access_list = []
            for access in issue.access:
                access_list.append(access.login)
            issue = self.startrek.issues[issue.key]
            issue.update(access=access_list + certificate_admins,
                         ignore_version_change=True)
