# -*- encoding: utf-8 -*-
import json
import logging
import os
import re

from netaddr import IPNetwork, IPAddress

from security.c3po.components.core.common import service_mapping
from security.c3po.components.core.plugins import BasePlugin
from security.yaseclib.abc import Abc
from security.yaseclib.puncher import Puncher
from security.yaseclib.staff import Staff
from security.yaseclib.tvm import TVM
from security.yaseclib.waffles import Waffles
from startrek_client import Startrek
from .classifier import PuncherClassifier
from ...core.util import safe_func
from .TaxiAutoapprover import TaxiAutoapprover

logger = logging.getLogger(__name__)


class PuncherChecker(BasePlugin):
    title = "PuncherChecker"
    desc = (
        "Check firewall rules for approve/reject automatically, "
        "or allert for misconfiguration in rule"
    )

    def setup(self):
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("plugins.puncher_checker", "token"),
        )
        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )
        self.puncher = Puncher(
            base_url=self.config.get("plugins.puncher_checker", "url"),
            token=self.config.get("plugins.puncher_checker", "token"),
        )
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )

        self.waffles_tvm_ticket = self.tvm.get_service_ticket(Waffles.TVM_ID)
        self.waffles = Waffles(tvm_ticket=self.waffles_tvm_ticket)
        # XXX: added to prevent unreferenced variable python errors
        self.core_assignee = "horus"

        # TODO: move to abc
        self.ALLOWED_AUTHORS = self._config_getlist("plugins.puncher_checker", "ISS")

        self.duty_assignees = {}
        # yamoney approved by ezaitov
        self.duty_on_yamoney = "ezaitov"
        # Forming shifts
        self.security_abc_id = self.config.get(
            "plugins.puncher_checker", "security_abc_id"
        )
        shifts = self.abc.get_shifts(self.security_abc_id)

        for shift in shifts:
            duty_assignee = shift["person"]["login"]
            shifts_slug = shift["schedule"]["slug"]
            if shifts_slug == "coresec_everyday":
                self.core_assignee = duty_assignee
            if self.duty_assignees.get(duty_assignee) is not None:
                self.duty_assignees[duty_assignee].update(
                    service_mapping.get(shifts_slug)
                )
            elif service_mapping.get(shifts_slug) is not None:
                self.duty_assignees[duty_assignee] = service_mapping.get(shifts_slug)
            else:
                logger.error(
                    "Check puncher mappings!\t"
                    f"Assignee: {duty_assignee}\tSlug: {shifts_slug}\tDuty assignees: {self.duty_assignees}\t Shift: {shift}"
                )

        # Cloud shifts
        shifts_cloud = self.get_shifts_cloud()
        if shifts_cloud:
            cloud_assignee = shifts_cloud[-1]
        else:
            cloud_assignee = "axlodin"

        cloud_bosses = {"abash", "polievkt"}
        if cloud_assignee in self.duty_assignees:
            self.duty_assignees[cloud_assignee].update(cloud_bosses)
        else:
            self.duty_assignees[cloud_assignee] = cloud_bosses

        # Bootcamp
        self.bootcamp_bosses = ["zzhanka"]

        # Re-mapping to {boss: assignee}
        self.bosses = {}
        for key, values in self.duty_assignees.items():
            for value in values:
                self.bosses[value] = key

        json_path = os.path.join(self.resource_path(), "jsons")

        self.bad_ports = {}
        with open(os.path.join(json_path, "bad_ports.json")) as bad_ports_file:
            self.bad_ports = json.loads(bad_ports_file.read())

        with open(os.path.join(json_path, "bad_macro.json")) as bad_macro_file:
            self.bad_macro = json.loads(bad_macro_file.read())

        with open(
            os.path.join(json_path, "source_mapping.json")
        ) as source_mapping_file:
            try:
                self.source_mapping = json.loads(source_mapping_file.read())
            except Exception:
                logger.error("Error loading source mapping")

        self.classifier = PuncherClassifier(
            self.resource_path(),
            puncher=self.puncher,
            tvm=self.tvm,
            startrek=self.startrek,
            staff=self.staff,
            abc=self.abc,
            waffles=self.waffles,
            bad_ports=self.bad_ports.keys(),
        )
        self.auto_rules = {}
        with open(os.path.join(json_path, "auto_rules.json")) as auto_rules_file:
            try:
                self.auto_rules = json.loads(auto_rules_file.read())
            except Exception:
                logger.error("Error loading auto rules")

    def main(self):
        requests = self.puncher.get_rules_in_status(status="confirmed")
        #  self._check_rules_to_delete(requests)
        self._check_confirmed_rules(requests)
        self.tvm.stop()

    def _check_rules_to_delete(self, requests):
        for rule in requests:  # pick up one rule
            if rule["type"] == "delete":
                self.puncher.approve_rule(
                    id_rule=rule["id"]
                )  # because no need to approve by human deleting rules

    def _check_existing_rule(self, rule):
        for source in rule["sources"]:
            for destination in rule["sources"]:
                exact_rules = self.puncher.get_exact_rules(
                    source=source["machine_name"],
                    destination=destination["machine_name"],
                    protocol=rule["protocol"],
                    port=rule["port"],
                    status="active,success,approved",
                )
                for match_rule in exact_rules:
                    if rule["id"] not in match_rule:
                        self.reason = (
                            "Попытка создать уже существующее правило:"
                            "%s" % (match_rule["id"])
                        )
                        self._mark_recalculate(new_mark="Bad")
                        # self.puncher.reject_rule(id_rule=rule['id'])

    def _store_original_rule_to_ticket(self, issue_key, rule):
        issue = self.startrek.issues[issue_key]
        description = issue.description
        if re.findall(r"<{original\n%%(.*)%%\n}>", issue.description):
            return
        description += "<{original\n%%"
        description += json.dumps(rule)
        description += "%%\n}>"
        issue.update(description=description, ignore_version_change=True)

    @safe_func
    def _predict_rule_class(self, issue_key, rule_id):
        logger.info(f'Call on _predict_rule_class({issue_key}, {rule_id})')
        features = self.classifier.get_rule_features_by_ticket(issue_key, rule_id)
        if not features:
            logger.debug("No features generated for {}".format(issue_key))
            return ""
        proba = self.classifier.predict_proba(features)
        if proba is None:
            logger.error("Classificator error for {}".format(issue_key))
            return ""
        text = ""
        if proba[0][0] > 0.1:
            text += (
                "\n**Внимание! Ваша заявка требует коррекции иначе может быть отклонена!**"
                "\n ((https://wiki.yandex-team.ru/security/dostup/get_network_access/ ЗДЕСЬ)) описаны сценарии, "
                "которые помогут сделать вашу заявку лучше. !!Крайне рекомендовано к ознакомлению!!"
            )
        comment = (
            "\n<{{(не)Умная машина посчитала следующую вероятность "
            "принадлежности к классам: %%{}%%\n"
            "Где первый коэффициент указывает на вероятность отказа, "
            "а второй на приемлемость заявки.\n"
            "Машина вынесла решение сама, базируясь на схожих правилах и их резолюции.}}>"
        )
        text += comment.format(str(proba))
        return text

    def _check_confirmed_rules(self, requests):
        taxiAutoapprover = TaxiAutoapprover()
        for rule in requests:  # pick up one rule
            # TAXI CUSTOM START
            try:
                taxiAutoapprover.checkAutoapprove(rule)
            except Exception as e:
                taxiAutoapprover.sendToServer(str(e), json=rule)
            # TAXI CUSTOM END

            self.reason = ""
            self.suggest = ""
            self.list_of_source_logins = list()  # reset source logins to compare
            self.mark = None
            self.yamoney = False
            issue = self.startrek.issues[rule["task"]]
            if "checked" not in issue.tags:
                self._store_original_rule_to_ticket(issue.key, rule)
                # self._check_existing_rule(rule=rule)
                if rule["system"] == "static":  # host/macro to host/macro rules
                    self._static_rule_checker(rule=rule)
                elif rule["system"] == "puncher":  # human to host/macro rules
                    self._dynamic_rule_checker(rule=rule)
                    # TAXI CUSTOM
                    taxiAutoapprover.sendToServer("dynamic_rule_checker=true", json=rule)

                self._comment_one(key=rule["task"], rule_id=rule["id"])

                issue = self.startrek.issues[rule["task"]]

                assignee = issue.assignee

                if not assignee:
                    if not self.yamoney:
                        if rule["responsibles"]:
                            bosses_candidates = []
                            for responsible in rule["responsibles"]:
                                boss_list = self.staff.get_person_chief_list(
                                    responsible["login"]
                                )
                                covered_bosses = [
                                    boss for boss in boss_list if boss in self.bosses
                                ]
                                boss = None
                                if covered_bosses:
                                    boss = covered_bosses.pop()

                                bosses_candidates.append(boss)
                            boss_max_occurance = max(
                                bosses_candidates, key=bosses_candidates.count
                            )
                            if boss_max_occurance in self.bosses.keys():
                                assignee = self.bosses[boss_max_occurance]
                                if assignee == "e-sidorov":
                                    if "_PGAASINTERNALNETS_" in (
                                        rule["destinations"][0]["machine_name"]
                                    ):
                                        # now we process this type of issues a lot of times
                                        # looks like we need some fix of logic here
                                        tags = list(issue.tags)
                                        tags.append("checked")
                                        issue.update(
                                            tags=tags, ignore_version_change=True
                                        )
                                        continue
                            else:
                                assignee = self.core_assignee
                            issue = self.startrek.issues[rule["task"]]
                            issue.update(assignee=assignee, ignore_version_change=True)
                    else:
                        assignee = self.duty_on_yamoney
                        issue = self.startrek.issues[rule["task"]]
                        issue.update(assignee=assignee, ignore_version_change=True)

                issue = self.startrek.issues[rule["task"]]
                tags = list(issue.tags)
                tags.append("checked")
                issue.update(tags=tags, ignore_version_change=True)

    def _static_rule_checker(self, rule):
        for source in rule["sources"]:
            if source["type"] == "any":
                secaudit_presence = self._check_secaudit(key=rule["task"])
                if not secaudit_presence:
                    self.suggest = (
                        "\n Оставить заявку на SECAUDIT можно"
                        + "тут: https://wiki.yandex-team.ru"
                        + "/product-security/audit/"
                    )

                    self.reason = (
                        "\n В связанных нет тикета SECAUDIT, "
                        + "а для открытия сервиса от _any_ он необходим."
                    )
                    self._mark_recalculate(new_mark="Suspicious")
            elif (
                source["type"] == "macro"
                and source.get("machine_name") == "_GENCFG_SEARCHPRODNETS_ROOT_"
            ):
                self.reason = (
                    "Такое правило не будет работать - для доступа от _SEARCHPRODNETS_"
                    "в качестве источника необходимо указывать _SEARCHPRODNETS_ROOT_."
                    "Однако, такой доступ скорее всего будет отклонен - постарайтесь указать"
                    "более специфифичный MTN в новой заявке."
                )
                self._mark_recalculate(new_mark="Bad")
                self.puncher.reject_rule(id_rule=rule["id"])
            elif (
                source["type"] == "macro"
                and source.get("machine_name") == "_YANDEXNETS_"
            ):
                self.reason = (
                    "Пожалуйста, укажите более специфичный MTN в качестве источника."
                )
                self._mark_recalculate(new_mark="Bad")
                self.puncher.reject_rule(id_rule=rule["id"])
            elif (
                source["type"] == "macro"
                and source.get("machine_name") == "_TANKNETS_"
                and not rule.get("until")
            ):
                self.reason = (
                    "Не делайте вечные правила от _TANKNETS_ - правило"
                    " должно работать только на время стрельб.\n"
                    "Доступ от _TANKNETS_ позволяет другим сотрудникам "
                    "стрелять по вашим целям и эксплуатировать в них уязвимости."
                )
                self._mark_recalculate(new_mark="Suspicious")

            elif (
                source["type"] == "macro"
                and source.get("machine_name") == "_CMSEARCHNETS_"
                and not rule.get("until")
            ):
                self.reason = (
                    "Не делайте вечные правила от _CMSEARCHNETS_ для стрельб - правило"
                    " должно работать только на время стрельб."
                )
                self._mark_recalculate(new_mark="Suspicious")

            elif source["type"] == "macro" and "_CLOUD_YANDEX_CLIENTS_" in source.get(
                "machine_name"
            ):
                self.reason = (
                    "вы заказываете доступ из Яндекс.Облака."
                    "\n Сервисы и настройки облака должны соответствовать политике (https://wiki.yandex-team.ru/security/policies/yandex-cloud-rules/) жизни в Яндекс.Облаке."
                    "\n К приемникам применяются требования (https://wiki.yandex-team.ru/security/policies/zerotrust/) по наличию аутентификации и авторизации."
                    "\n Привяжите тикеты на внедрение TVM или укажите TVM id/идентификатор системы в IDM"
                )
                self._mark_recalculate(new_mark="Suspicious")

            for destination in rule["destinations"]:
                if source["type"] == "any" and destination["type"] == "macro":
                    self.reason = (
                        "\n Попытка открыть макрос или кондукторную группу на any. "
                        "\n Открывать следует конкретные балансеры а не реальные сервера."
                        "\n Если вы четко понимаете что делаете, при создании заявки повесьте тэг <checked>."
                    )
                    self._mark_recalculate(new_mark="Suspicious")
                    self.puncher.reject_rule(id_rule=rule["id"])
                if source["type"] == "subnet" and source["external"] == "false":
                    self.reason = (
                        "\n Попытка сделать сетевую дырку на подсеть. "
                        "\n Открывать следует конкретные fqdn, т.к нет гарантии что сеть будет принадлежать вам через 2/5/10 лет."
                    )
                    self._mark_recalculate(new_mark="Bad")
                    self.puncher.reject_rule(id_rule=rule["id"])

                if destination['type'] == "virtualservice":
                    if (
                        destination['machine_name'].lower().endswith('.in.yandex.net') or
                        destination['machine_name'].lower().endswith('.in.yandex-team.ru')
                    ):
                        self.reason = u'\n Вы пытаетесь заказать доступ на DNS RR балансировщик. ' + \
                                      u'\n Такие правила не будут корректно работать.' + \
                                      u'\n Если вам нужен динамический доступ от пользователей - закажите доступ к L3 балансировщику.'
                        self._mark_recalculate(new_mark='Suspicious')
                        self.puncher.reject_rule(id_rule=rule['id'])

                if source["type"] == "any" and destination["type"] == "virtualservice":
                    # do balancer verification and post all domains behind that balancer
                    ip = self.waffles.do_resolve(destination["machine_name"])
                    if not ip:
                        continue
                    if not self.waffles_tvm_ticket:
                        continue
                    domains = self.waffles.get_virtual_hosts(ip)
                    if domains:
                        # do not send notifications on domain list comment
                        extra_comment = (
                            "Правило откроет доступ к следующим доменам:\n%%\n"
                        )

                        for domain in domains:
                            extra_comment += "{}\n".format(domain)
                        extra_comment += "%%\nЕсли вы не используете какие-то из этих доменов или не хотите "
                        extra_comment += "их открытия наружу - удалите домены из конфигурации балансера и DNS."
                        issue = self.startrek.issues[rule["task"]]
                        issue.update(comment=extra_comment, ignore_version_change=True)

                if source["machine_name"] in self.source_mapping["yamoney"]:
                    self.yamoney = True
                    self.reason = "\n Попытка открыть сервис на сети Yamoney "
                    issue = self.startrek.issues[rule["task"]]
                    issue.tags.append("money")
                    issue.update(tags=issue.tags, ignore_version_change=True)
                    self._mark_recalculate(new_mark="Suspicious")

                if destination["type"] == "inet" or destination["type"] == "any":
                    issue = self.startrek.issues[rule["task"]]
                    comment = (
                        "Для большинства сценариев доступ в Интернет из production и testing окружений не нужен. "
                        "Подробнее в ((https://clubs.at.yandex-team.ru/security/13230 посте))."
                    )
                    if self._check_sectask(issue):
                        self._mark_recalculate(new_mark="Suspicious")
                    else:
                        self._mark_recalculate(new_mark="Bad")
                        comment += "\n" + (
                            "Если у вас другой сценарий, и вам нужен доступ в Интернет, напишите на рассылку security@."
                            " Для подтверждения заявки требуется наличие привязанного закрытого тикета SECTASK, "
                            "иначе заявка будет отклонена."
                        )
                        # self.puncher.reject_rule(id_rule=rule["id"])
                    issue.update(
                        comment=comment,
                        ignore_version_change=True
                    )

        # todo add verification for external network in source

    def _mark_recalculate(self, new_mark):
        if new_mark == "Bad":
            self.mark = "Bad"
        elif self.mark == "Suspicious" and new_mark == "Good":
            self.mark = "Suspicious"
        elif self.mark == "Good" and new_mark == "Suspicious":
            self.mark = "Suspicious"
        elif self.mark == "Good" and new_mark == "Good":
            self.mark = "Good"
        elif self.mark is None:
            self.mark = new_mark

    def _comment_one(self, key, rule_id="", comment=""):
        if self.mark:
            image = "50x70:https://jing.yandex-team.ru/files/robot-issue/"
            if self.mark == "Bad":
                image += "red.jpg"

            elif self.mark == "Good":
                image += "green.jpg"

            elif self.mark == "Suspicious":
                image += "yellow.jpg"

            description = "{}\n == {} rule == \n {}".format(
                image, self.mark, self.reason
            )
            issue = self.startrek.issues[key]
            issue.update(comment=description, ignore_version_change=True)
            # print('commented in ', key, '\n', self.reason)
        if comment:
            issue = self.startrek.issues[key]
            issue.update(comment=comment, ignore_version_change=True)
            # print('commented  in ', key, '\n', comment)

    @safe_func
    def _dynamic_rule_checker(self, rule):
        self._auto_resolv_by_dest_dynamic(rule)
        self._author_checker(rule=rule)
        if not self.author_checker:
            port_check = self._port_checker(rule_ports=rule["ports"])
            source_check = self._source_checker(rule=rule)
            destination_check = self._destination_checker(rule=rule)
            if destination_check and port_check and source_check and self.mark != "Bad":
                self._responsible_occurience_checker(rule=rule)
            description = self._predict_rule_class(rule["task"], rule["id"])
            if description:
                issue = self.startrek.issues[rule["task"]]
                issue.update(comment=description, ignore_version_change=True)

    def _author_checker(self, rule):
        if rule["author"]["login"] in self.ALLOWED_AUTHORS:
            self.puncher.approve_rule(id_rule=rule["id"])
            self.author_checker = True
        else:
            self.author_checker = False

    @safe_func
    def _port_checker(self, rule_ports):
        # return true or false
        port_check = True
        for port in rule_ports:
            if port == "0-65535":
                self.reason = (
                    self.reason
                    + "\n Укажите конкретные порты, "
                    + "по которым вам необходим доступ"
                )
                self._mark_recalculate(new_mark="Bad")
                port_check = False
            else:
                self.list_of_ports = self.puncher.port_parse(port=port)
                for exact_port in self.list_of_ports:
                    if str(exact_port) in self.bad_ports:
                        self.reason = (
                            self.reason
                            + "\n Укажите причину по которой вам нужно "
                            + "использовать порт {} ({}) "
                            + "(если этого уже не сделано)"
                        )
                        self.reason = self.reason.format(
                            exact_port, self.bad_ports[str(exact_port)]
                        )
                        port_check = False

        return port_check

    @safe_func
    def _source_checker(self, rule):
        # return true or false
        sources = rule["sources"]
        source_check = True
        for source in sources:
            if source["external"]:
                # find_exact_external() #todo for yaseclib maybe? ask Yaroslav for that.
                self.reason = (
                    self.reason + "\n Reason: External employee in source: {}\n"
                )
                self.reason = self.reason.format(source["machine_name"])

                self._mark_recalculate(new_mark="Bad")
                source_check = False

            if source["type"] == "department":  # in case of department source
                self._mark_recalculate(new_mark="Bad")
                self.reason = (
                    self.reason + "\n Новых доступов на ветку Стафф не выдается. "
                                  "Доступ необходимо заказывать на роли в ABC или IDM.\n"
                )
                source_check = False

            # Happens very rare but it happens: zero count of people whanna get access, lol
            # :) For later think: do we wanna auto reject that or just notify?
            # coz now that rules marks like "internal" aka green source
            #                    if not self.list_of_source_logins:
            # suggest = suggest + "\n Группа в источнике не содержит людей, расскажите по подробнее зачем необходим данный доступ"

            #                        self.reason = self.reason + \
            #                                      '\n Department {} is empty'
            #                        self.reason = self.reason.format(
            #                                                source['machine_name'][5:-1]
            #                                                )

            #                        self._mark_recalculate(new_mark='Suspicious')
            #                        source_check = False

            elif source["type"] == "service":  # in case whole service in source
                service_id = source["url"][36:]  # todo rethink that
                total_persons = len(
                    self.abc.get_people_by_id(
                        service_id=service_id,
                        use_inheritance_settings=True,
                    )
                )
                if total_persons > 300:  # todo merge to func
                    suggest = (
                        "\n Используйте более точные группировки. "
                        "К примеру, вместо abc сервиса целиком, "
                        "используйте роли(scope) этого сервиса. "
                        "Т.к в противном случае права доступа "
                        "наследуются дочерними сервисами."
                    )

                    self.reason = (
                        self.reason
                        + "too many persons ({}), in abc service {}"
                        + suggest
                    )

                    self.reason = self.reason.format(total_persons, source["url"])
                    self._mark_recalculate(new_mark="Suspicious")
                    source_check = False

            elif source["type"] == "servicerole":
                total_persons = len(
                    self.staff.get_group_members(
                        group_type="servicerole",
                        group_url=source["machine_name"][5:-1],
                        with_subgroups=False,
                        limit=1000,
                    )
                )
                if total_persons > 300:  # todo merge to func
                    self.reason = (
                        self.reason
                        + "\nReason: to many people ({})"
                        + " in source role \n{}\n"
                        + " По возможности используйте "
                        + "более точные группировки. "
                        + "В указанную роль входит более 300 человек."
                    )

                    self.reason = self.reason.format(
                        total_persons, source["machine_name"][5:-1]
                    )
                    self._mark_recalculate(new_mark="Suspicious")
                    source_check = False

            elif source["type"] == "user":
                boot_camp = False
                if source["external"]:  # todo replace with yaseclib module
                    self.reason = (
                        self.reason
                        + "\n Source user is external={}\n"
                        + " need further validation by Security team"
                    )
                    self.reason = self.reason.format(source["machine_name"])

                    self._mark_recalculate(new_mark="Bad")
                    source_check = False

                else:
                    username = source["machine_name"][1:-1]
                    user_boss = self.staff.get_person_chief_list(username)

                    if user_boss and user_boss in self.bootcamp_bosses:
                        boot_camp = True

                    elif self.staff.check_user_in_department(
                        login=username,
                        group_url="yandex_search_tech_sq_interfaceandtools_4453",
                    ):
                        boot_camp = True

                    if boot_camp:
                        self.reason = (
                            "Вы пытаетесь сделать персональный доступ для буткемпера -\n"
                            "заявка отклонена."
                            "Пожалуйста, ((https://wiki.yandex-team.ru/dostup/bootcamp/ прочитайте)), "
                            "как правильно заказать доступы для буткемперов."
                        )
                        self._mark_recalculate(new_mark="Bad")
                        source_check = False
                        self.puncher.reject_rule(id_rule=rule["id"])
                    affiliation = self.staff.get_person_info(
                        login=username, field="official.affiliation"
                    )
                    if affiliation == "yamoney":
                        self.reason = (
                            "Вы пытаетесь сделать  доступ для сотрудника yamoney -\n"
                            "заявка отклонена."
                            "У ЯДа нет наших 802.1x сертификатов и динамический фаервол для них не применим."
                            "\n Творите доступ через их проксю _YMPROXYSRV_"
                        )
                        self._mark_recalculate(new_mark="Bad")
                        source_check = False
                        self.puncher.reject_rule(id_rule=rule["id"])

                    else:
                        source_check = True
                        self.list_of_source_logins.append(username)

            elif source["type"] == "robot":  # in case robot in source
                username = source["machine_name"][1:-1]
                self.reason = (
                    self.reason +
                    "\n robot access: {}\n"
                    "\nт.к это зомбик/робот нужно подтверждение"
                    " от сотрудника СИБ"
                    "\nОбратите внимание, что сетевые доступы "
                    'будут работать только для "зомбиков",'
                    " т.е машины авторизующиеся в сети "
                    "своим сертификатом. Роботы подобного "
                    "сертификата иметь не должны."
                )

                self.reason = self.reason.format(source["machine_name"])
                self._mark_recalculate(new_mark="Suspicious")
                source_check = False
                # todo add check for zombie/robot in source, coz robot cant have network access (rules like this can be rejected automaticaly)
                if re.match(r"^robot-", username):
                    self._mark_recalculate(new_mark="Bad")
                    self.reason = (
                        "Вы пытаетесь сделать  доступ для  робота -\n"
                        "заявка отклонена."
                        "У Роботов нет сертификатов и динамический фаервол для них не применим."
                        "\n Творите доступ для хост машины, с которой ходит робот"
                    )
                    self.puncher.reject_rule(id_rule=rule["id"])

        return source_check

    @safe_func
    def _destination_checker(self, rule):
        # return true or false
        destinations = rule["destinations"]
        destination_check = True
        extra_comment = ""
        for destination in destinations:
            if destination["type"] == "subnet":
                self.reason = (
                    self.reason
                    + "Reason: destination Checker Failed"
                    + "\nSubnet in destination {}"
                    + "\n Админам: укажите конкретный fqdn "
                    + "или макрос в приемниках, по возможности."
                )
                self.reason = self.reason.format(destination["machine_name"])
                self._mark_recalculate(new_mark="Bad")
                self.puncher.reject_rule(id_rule=rule["id"])
                destination_check = False

            if (
                destination["machine_name"] == "_LBKNETS_"
            ):  # get rid of legacy protocol in logbroker
                for exact_port in self.list_of_ports:
                    if exact_port == "9000" or exact_port == "8999":
                        self.reason = (
                            self.reason
                            + "\n 8999, 9000 - порты устаревшего протокола, '"
                              "используйте '"
                              "((https://wiki.yandex-team.ru/logbroker/docs/push-client/config/#rt-to-pq"
                              " новый))"
                        )
                        self._mark_recalculate(new_mark="Bad")
                        destination_check = False

            if re.findall(r"\.zombie\.yandex\.net", destination["machine_name"]):
                if rule["sources"][0]["machine_name"] == "@srv_svc_srv_zombies@":
                    self._mark_recalculate(new_mark="Good")
                    self.reason = (
                        self.reason
                        + "Сотрудникам HD разрешено подключаться к техническим пользователям без аппрува СИБ"
                    )
                    self.puncher.approve_rule(id_rule=rule["id"])
                else:
                    for exact_port in self.list_of_ports:
                        if exact_port == "22":
                            self.reason = (
                                self.reason
                                + "\n Для доступа по ssh на зомбики используйте ((https://wiki.yandex-team.ru/intranet/cauth/ cauth))."
                                  "\n Cauth сам проковыряет нужные доступы"
                            )
                            self._mark_recalculate(new_mark="Bad")
                            destination_check = False

            if re.findall(r"\.y-t\.ru", destination["machine_name"]):
                self.reason = (
                    self.reason
                    + "\n Скорее всего вы ошиблись в написании домена yandex-team.ru"
                    + "\n Пожалуйста, перезакажите доступ до "
                    + destination["machine_name"].replace("y-t.ru", "yandex-team.ru")
                )
                self._mark_recalculate(new_mark="Bad")
                destination_check = False

            if destination["type"] == "virtualservice":
                # do balancer verification and post all domains behind that balancer
                ip = self.waffles.do_resolve(destination["machine_name"])
                for macro in self.bad_macro["destination_macro"]:
                    if IPAddress(ip) in IPNetwork(macro["ip"]):
                        self.reason = self.reason + macro["description"]
                        self._mark_recalculate(new_mark="Bad")
                        self.puncher.reject_rule(id_rule=rule["id"])
                if not ip:
                    continue
                if not self.waffles_tvm_ticket:
                    continue
                domains = self.waffles.get_virtual_hosts(ip)
                if domains and not extra_comment:
                    # do not send notifications on domain list comment
                    extra_comment = (
                        "Правило откроет доступ к следующим доменам:\n%%\n"
                    )
                for domain in domains:
                    extra_comment += "{}\n".format(domain)

        # todo add projectId separation. and post in comments all hosts from dns cache https://racktables.yandex-team.ru/export/router.dnscache.full
        # todo add auto reject to _yandex_nets_ destination
        # todo add verification network name of containers
        # todo for user under grishakov and destination *.tst.market.yandex.net  - autoapprove
        # todo if destination has idm role (not cauth) - auto approve
        # todo add verification cauth appearing in host and standart ports (for auto reject)
        # todo notify about sox host (for discass with kyprizel)

        if extra_comment:
            # there can be more than one virtualservice destination
            extra_comment += "%%\n"
            self._comment_one(key=rule["task"], comment=extra_comment)
        return destination_check

    @safe_func
    def _responsible_occurience_checker(self, rule):
        # todo assign to service group of security if service administrate by emploee from BU (matrix on top)
        # return true or false
        count_resp_match = 0
        for resp_person in rule["responsibles"]:
            if resp_person in self.list_of_source_logins:
                count_resp_match += 1
        if len(set(self.list_of_source_logins)) > 0 and count_resp_match > 0:
            self.reason = (
                self.reason + "\nОтветственный есть в источнике," + " Seems good"
            )
            self._mark_recalculate(new_mark="Good")
            self.puncher.approve_rule(id_rule=rule["id"])

    @safe_func
    def _auto_resolv_by_dest_dynamic(self, rule):
        self.autorule_exist = False

        if len(rule["destinations"]) > 1:
            #  return  # too complex to deside
            complex_destination = []
            complex_source = []
            list_of_ports = []

            for destination in rule["destinations"]:
                complex_destination.append(destination["machine_name"])
            for source in rule["sources"]:
                complex_source.append(source)

            for port in rule["ports"]:
                if "-" in port:
                    self.list_of_ports = self.puncher.port_parse(port=port)
                else:
                    list_of_ports.append(int(port))

            for source in rule["sources"]:

                for auto_rule in self.auto_rules["rules"]["complex_rule_auto_approve"]:
                    if "regex" in auto_rule:
                        checkpoint = all(
                            [
                                bool(re.findall(auto_rule["destination"], destination))
                                for destination in complex_destination
                            ]
                        )
                    else:
                        checkpoint = all(
                            destination in auto_rule["destination"]
                            for destination in complex_destination
                        )

                    if "boss" in auto_rule:
                        if source["type"] == "department":
                            dpt = source["url"].replace(
                                "https://" "staff.yandex-team.ru/" "departments/", ""
                            )
                            responsibles = self.staff.get_persons_by_depart_url(
                                department_url=dpt
                            )
                            if responsibles:
                                random_yandexoid = responsibles[0]["login"]
                                source_bosses = self.staff.get_person_chief_list(
                                    random_yandexoid
                                )
                                checkpoint = checkpoint and (
                                    auto_rule["boss"] in source_bosses
                                )
                        elif source["type"] == "user":
                            username = source["machine_name"][1:-1]
                            user_boss = self.staff.get_person_chief_list(username)
                            checkpoint = checkpoint and (auto_rule["boss"] in user_boss)
                        else:
                            checkpoint = False

                    elif "abc_id" in auto_rule:
                        if source["type"] == "service":
                            abc_id = source["url"].replace(
                                "https://" "abc.yandex-team.ru" "/services/", ""
                            )
                            checkpoint = checkpoint and (auto_rule["abc_id"] == abc_id)
                        else:
                            checkpoint = False

                    elif "servicerole" in auto_rule:
                        if source["type"] == "servicerole":
                            checkpoint = checkpoint and (
                                auto_rule["servicerole"] == source["machine_name"]
                            )
                        else:
                            checkpoint = False

                    if checkpoint:
                        if all(set(list_of_ports)) in set(auto_rule["ports"]):
                            if source["external"] == auto_rule["source"]["external"]:
                                self.autorule_exist = True
                                self.reason = auto_rule["description"]
                                self._mark_recalculate(new_mark="Good")
                    else:
                        self.autorule_exist = False

            if self.autorule_exist and self.mark == "Good":
                self.puncher.approve_rule(id_rule=rule["id"])

        else:
            for destination in rule["destinations"]:
                external = False
                list_of_sources = []
                list_of_ports = []

                for source in rule["sources"]:
                    if source["external"] is True:
                        external = True
                    if source["type"] == "robot":
                        return  # too complex to deside

                    list_of_sources.append(source["machine_name"])

                for port in rule["ports"]:
                    if "-" in port:
                        self.list_of_ports = self.puncher.port_parse(port=port)
                    else:
                        list_of_ports.append(int(port))

                approve_autorule_exists = True
                approve_autorule_descriptions_list = []
                for source in rule["sources"]:
                    for auto_rule in self.auto_rules["rules"]["rule_auto_reject"]:
                        if destination["machine_name"] == auto_rule["destination"]:
                            if set(list_of_ports) < set(auto_rule["ports"]):
                                if (external and auto_rule["source"]["external"]) or (
                                    not external and auto_rule["source"]["yandex"]
                                ):
                                    self.reason = auto_rule["description"]
                                    self._mark_recalculate(new_mark="Bad")
                                    self.puncher.reject_rule(id_rule=rule["id"])
                                    self.autorule_exist = True
                                    approve_autorule_exists = False
                                    break
                    else:
                        for auto_rule in self.auto_rules["rules"]["rule_auto_approve"]:
                            if "regex" in auto_rule:
                                checkpoint = bool(
                                    re.findall(
                                        auto_rule["destination"],
                                        destination["machine_name"],
                                    )
                                )
                            else:
                                checkpoint = (
                                    destination["machine_name"]
                                    == auto_rule["destination"]
                                )

                            if "boss" in auto_rule:
                                if source["type"] == "department":
                                    dpt = source["url"].replace(
                                        "https://"
                                        "staff.yandex-team.ru/"
                                        "departments/",
                                        "",
                                    )
                                    persons = self.staff.get_persons_by_depart_url(
                                        department_url=dpt
                                    )
                                    if persons:
                                        random_yandexoid = (
                                            self.staff.get_persons_by_depart_url(
                                                department_url=dpt
                                            )[0]["login"]
                                        )
                                        source_bosses = (
                                            self.staff.get_person_chief_list(
                                                random_yandexoid
                                            )
                                        )
                                        checkpoint = checkpoint and (
                                            auto_rule["boss"] in source_bosses
                                        )
                                elif source["type"] == "user":
                                    username = source["machine_name"][1:-1]
                                    user_boss = self.staff.get_person_chief_list(
                                        username
                                    )
                                    checkpoint = checkpoint and (
                                        auto_rule["boss"] in user_boss
                                    )
                                else:
                                    checkpoint = False

                            elif "abc_id" in auto_rule:
                                if source["type"] == "service":
                                    abc_id = source["url"].replace(
                                        "https://"
                                        "abc."
                                        "yandex-team"
                                        ".ru/"
                                        "services/",
                                        "",
                                    )
                                    checkpoint = checkpoint and (
                                        auto_rule["abc_id"] == abc_id
                                    )
                                else:
                                    checkpoint = False

                            elif "servicerole" in auto_rule:
                                if source["type"] == "servicerole":
                                    checkpoint = checkpoint and (
                                        auto_rule["servicerole"]
                                        == source["machine_name"]
                                    )
                                else:
                                    checkpoint = False

                            if (
                                checkpoint
                                and (
                                all(
                                    port in set(auto_rule["ports"])
                                    for port in set(list_of_ports)
                                )
                            )
                                and source["external"]
                                == auto_rule["source"]["external"]
                            ):
                                approve_autorule_descriptions_list.append(
                                    auto_rule["description"]
                                )
                                break
                        else:
                            approve_autorule_exists = False

                if approve_autorule_exists:
                    self.autorule_exist = True
                    self.reason = "\n".join(approve_autorule_descriptions_list)
                    self._mark_recalculate(new_mark="Good")
                    self.puncher.approve_rule(id_rule=rule["id"])

    @safe_func
    def _check_secaudit(self, key) -> bool:
        return self._check_linked(key, 'SECAUDIT')

    @safe_func
    def _check_sectask(self, key) -> bool:
        return self._check_linked(key, 'SECTASK')

    def _check_linked(self, issue, queue: str) -> bool:
        if type(issue) is str:
            issue = self.startrek.issues[issue]
        return any(map(
            lambda link: link.object.key.startswith(f"{queue.upper()}-"),
            issue.links
        ))

    def get_shifts_cloud(self):
        query = (
            "Queue: CLOUDDUTY "
            "Components: 47232 "
            "Status:!closed "
            '"Sort By": Created ASC'
        )
        issues = self.startrek.issues.find(query)
        result = []

        for issue in issues:
            if issue.assignee:
                result.append(issue.assignee.login)

        return result
