# -*- coding: utf-8 -*-
import datetime
import logging
import os
import random
import re
import urllib
from collections import defaultdict

from jinja2 import Environment, FileSystemLoader

from security.c3po.components.core.common import boss_followers_mapping
from security.c3po.components.core.plugins import BasePlugin
from startrek_client import Startrek
from security.yaseclib.abc import Abc
from security.yaseclib.gap import Gap
from security.yaseclib.impulse import ImpulseQueuer
from security.yaseclib.molly import Molly
from security.yaseclib.staff import Staff
from security.yaseclib.tvm import TVM


# Too huge for now
# flake8: noqa
# TODO: Refactor

class Secaudit(BasePlugin):
    title = u"SECAUDIT"
    desc = u"Работа с заявками на SECAUDIT в одноимённой очереди. Форма: https://forms.yandex-team.ru/admin/309"

    def setup(self):
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(
            Gap.CALENDAR_YATEAM_TVM_ID
        )

        self.gap = Gap(
            base_url=self.config.get("gap", "url"),
            token=self.config.get("gap", "token"),
            base_calendar_url=self.config.get("calendar", "url"),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket,
        )

        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.molly = Molly(
            base_url=self.config.get("molly", "url"),
            token=self.config.get("molly", "token"),
        )

        self.env = Environment(
            loader=FileSystemLoader(self.resource_path() / "templates")
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.impulse = ImpulseQueuer(
            base_url=self.config.get("impulse", "url"),
            token=self.config.get("impulse", "token"),
        )

        self.service_mapping = {
            "a-abakumov": {"lyadzhin": 64058},
            "ilyaon": {"lyadzhin": 64058},
            "aleksei-m": {"dpolischuk": 88935},
            "axlodin": {"abash": 51405},
            "gpwn": {"abash": 51405},
            "cheremushkin": {"abash": 51405},
            "kaleda": {
                "ashevenkov": 64055,
                "tumryaeva": 64055,
                "atarabrin": 64055,
                "evzvonkova": 64055,
                "mlevin": 64055,
                "abalakhnin": 64055,
                "a-pianov": 64055,
                "pavel-b": 64055,
                "knepom": 64055,
                "vlikh": 64055,
                "geradmi": 64055,
                "daria-rezinko": 64055,
                "liumin": 64055,
                "fantamp": 64055,
                "akostin": 64055
            },
            "ya-andrei": {
                "ashevenkov": 64055,
                "tumryaeva": 64055,
                "atarabrin": 64055,
                "evzvonkova": 64055,
                "mlevin": 64055,
                "abalakhnin": 64055,
                "a-pianov": 64055,
                "pavel-b": 64055,
                "knepom": 64055,
                "vlikh": 64055,
                "geradmi": 64055,
                "daria-rezinko": 64055,
                "liumin": 64055,
                "fantamp": 64055,
                "akostin": 64055
            },
            "ichalykin": {
                "ashevenkov": 64055,
                "tumryaeva": 64055,
                "atarabrin": 64055,
                "evzvonkova": 64055,
                "mlevin": 64055,
                "abalakhnin": 64055,
                "a-pianov": 64055,
                "pavel-b": 64055,
                "knepom": 64055,
                "vlikh": 64055,
                "geradmi": 64055,
                "daria-rezinko": 64055,
                "liumin": 64055,
                "fantamp": 64055,
                "akostin": 64055
            },
            "enr1g": {"rommich": 64056},
            "rshemyakin": {"asavinovsky": 64061},  # Медиа
            "azizalimov": {"daniilsh": 64057},
            "audap": {"daniilsh": 64057},
            "naumov-al": {
                "daniilsh": 64057,
                "golubtsov": 31293,
                "shibanov": 31293
            },
            "dselianin": {
                "golubtsov": 31293,
                "shibanov": 31293
            },
            "pavkir": {
                "aleshin": 64059,
                "anton-fr": 64060  # Дзен
            },
            "ezzer": {
                "aleshin": 64059,
                "anton-fr": 64060  # Дзен
            },
            "procenkoeg": {
                "agodin": 17661,
                "fomichev": 17661,
                "babenko": 17661,
                "chubinskiy": 17661,
                "milovidov": 17661,
                "bkht": 17661,
                "tsufiev": [74524],
            },
            "anton-k": {
                "dshtan": 17661,
                "agodin": 17661,
                "fomichev": 17661,
                "babenko": 17661,
                "chubinskiy": 17661,
                "milovidov": 17661,
                "bkht": 17661,
                "tsufiev": [74524],
            },
            "shikari": {
                "dshtan": 17661,
                "agodin": 17661,
                "fomichev": 17661,
                "babenko": 17661,
                "chubinskiy": 17661,
                "milovidov": 17661,
                "bkht": 17661,
                "tsufiev": [74524],
            },
            "ezaitov": {
                "dshtan": 17661,
                "agodin": 17661,
                "fomichev": 17661,
                "babenko": 17661,
                "chubinskiy": 17661,
                "milovidov": 17661,
                "bkht": 17661,
                "tsufiev": [74524],
            },
            "melkikh": {
                "dshtan": 17661,
                "agodin": 17661,
                "fomichev": 17661,
                "babenko": 17661,
                "chubinskiy": 17661,
                "milovidov": 17661,
                "bkht": 17661,
                "tsufiev": [74524],
            },
            "buglloc": {
                "dshtan": 17661,
                "agodin": 17661,
                "fomichev": 17661,
                "babenko": 17661,
                "chubinskiy": 17661,
                "milovidov": 17661,
                "bkht": 17661,
                "tsufiev": [74524],
            },
            "dbeltukov": {
                "aidev": 104558,
            },
            "mprokopovich": {
                "aidev": 104558,
            }
        }

        self.sersec = defaultdict(dict)
        for key, values in self.service_mapping.items():
            for value in values.keys():
                self.sersec[value][key] = values[value]
        self.antifraud_team = None
        try:
            self.antifraud_team = self._config_getlist("secaudit", "antifraud_team")
        except:
            pass

    def main(self):
        errors = []
        for issue in self.get_unchecked_issues():
            logging.info("Processing issue {}".format(issue.key))
            try:
                self._process_issue(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't procces task.")
            try:
                self._auto_approve(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't check if task should be auto approved.")

        for issue in self.get_issues_for_assign():
            logging.info("Processing issue {}".format(issue.key))
            try:
                self._auto_assign(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't auto assign the task.")
        self.tvm.stop()

    def _process_issue(self, issue):
        errors = []
        abc_id = 0

        # Force the form
        if not (
            self._is_valid_issue_origin(issue) and self._is_correct_description(issue)
        ):
            return

        # Normalize issue
        try:
            issue = self.normalize_issue(issue)
        except Exception as e:
            with self.sentry_module.push_scope():
                self.sentry_module.capture_exception()
            errors.append("Can't normalize issue. %s." % str(e))

        if get_issue_field(issue, field=u"Сборка браузера"):
            st_queue = "BROWSER"
        else:
            st_queue = self._get_issue_st_queue(issue)

        if not st_queue:
            self._set_checked(issue, errors)
            return

        if issue.abcService:
            abc_id = int(issue.abcService[0].as_dict()["id"])

        # Initiate infrasec audit
        if self.config.getboolean("plugins.secaudit", "init_infra_audit") and not nirvana_check(
                    issue):
            try:
                self._init_infra_audit(issue)
            except Exception:
                with self.sentry_module.push_scope():
                    self.sentry_module.capture_exception()
                errors.append("Can't initiate infrasec audit.")

        # Initiate scan by Molly
        if self.config.getboolean("plugins.secaudit", "init_molly_scan") and not nirvana_check(
                    issue):
            try:
                self._init_molly_scan(issue, st_queue, abc_id)
            except Exception as e:
                with self.sentry_module.push_scope() as scope:
                    self.sentry_module.capture_exception()
                logging.exception("Molly init_scan returned False")
                errors.append(
                    "Can't initiate scan by Molly (%s). %s." % (abc_id, str(e))
                )

        # Force Molly control (add task for integration)
        if self.config.getboolean("plugins.secaudit", "force_molly") and not nirvana_check(
                    issue):
            try:
                self._force_molly_control(issue, st_queue)
            except Exception:
                with self.sentry_module.push_scope() as scope:
                    self.sentry_module.capture_exception()
                errors.append("Can't add task for Molly control.")

        # Intialize repository scan by AppChecker
        if self.config.getboolean("plugins.secaudit", "init_impulse_repo_scan"):
            if nirvana_check(issue):
                # TODO: new Impuls' scanner type
                pass
            else:
                try:
                    self._init_impulse_repo_scan(issue, st_queue)
                except Exception as e:
                    with self.sentry_module.push_scope() as scope:
                        self.sentry_module.capture_exception()
                    logging.exception("impulse.init_impulse_repo_scan returned False")
                    errors.append("Can't initialize repository scan. {}".format(e))

        # Force CSP (add task for integration)
        if self.config.getboolean("plugins.secaudit", "force_csp") and not nirvana_check(
                    issue):
            try:
                self._force_csp(issue, st_queue)
            except Exception:
                with self.sentry_module.push_scope() as scope:
                    self.sentry_module.capture_exception()
                errors.append("Can't add task for CSP control.")

        # Force TVM (add task for integration)
        if self.config.getboolean("plugins.secaudit", "force_tvm"):
            try:
                self._force_tvm(issue, st_queue)
            except Exception:
                with self.sentry_module.push_scope() as scope:
                    self.sentry_module.capture_exception()
                errors.append("Can't add task for TVM control.")

        # Force certificate pinning (add task for integration)
        if self.config.getboolean("plugins.secaudit", "force_pinning"):
            try:
                self._force_pinning(issue, st_queue)
            except Exception:
                with self.sentry_module.push_scope() as scope:
                    self.sentry_module.capture_exception()
                errors.append("Can't add task for certificate pinning control.")

        if self.antifraud_team:
            try:
                self._summon_antifraud(issue)
            except Exception:
                with self.sentry_module.push_scope() as scope:
                    self.sentry_module.capture_exception()
                errors.append("Can't summon antifraud team.")

        # Auto assign
        try:
            self._auto_assign(
                issue,
                first_line=self.config.getboolean("plugins.secaudit", "first_line"),
                abc_id=abc_id,
            )
        except Exception:
            with self.sentry_module.push_scope():
                self.sentry_module.capture_exception()
            errors.append("Can't auto assign the task.")

        metrics = dict()
        answers = self._get_answers(issue)
        risk_level = self._calculate_risk_level(metrics, answers)
        self._set_checked(issue, metrics, answers, risk_level, errors)

        # Post actions
        if self.config.getboolean("language_notify", "enabled"):
            self._language_notify(issue)

    def _init_impulse_repo_scan(self, issue, st_queue):
        repo_urls = self.get_repo_urls(issue)
        if repo_urls:
            repositories = []
            include_patterns = []
            exclude_patterns = ["test", "mocks"]
            for repo_url in repo_urls:
                repositories.append(
                    {"url": repo_url.get("url", ""), "branch": repo_url.get("branch")}
                )
                if repo_url.get("add_path"):
                    include_patterns.append(repo_url.get("add_path"))
            parameters = {
                "repositories": repositories,
                "include_patterns": include_patterns,
                "exclude_patterns": exclude_patterns,
            }
            self.impulse.send_to_impulse(
                issue=issue,
                org_id=self.config.getint("impulse", "org_id"),
                parameters=parameters,
                callback_url=self.config.get("impulse", "callback_url"),
                tracker_queue=st_queue,
            )

    def _init_molly_scan(self, issue, st_queue, abc_id):
        testing_urls = self._get_testing_urls(issue)
        if not testing_urls:
            msg = "Please, provide testing URL! Molly init_scan returned False."
            raise Exception(msg.format(testing_urls, issue.key))
        for testing_url in testing_urls:
            if not testing_url:
                continue
            scan_id = self.molly.init_scan(
                testing_url, st_queue=st_queue, st_ticket=issue.key, abc_id=abc_id
            )
            if scan_id:
                msg = "Molly scan scan_id: {}, testing_url: {}, ticket: {}"
                logging.info(msg.format(scan_id, testing_url, issue.key))
            else:
                msg = "Molly init_scan returned False for URL {}. Ticket {}"
                raise Exception(msg.format(testing_url, issue.key))

    def _language_notify(self, issue):
        other_languages = get_issue_field(issue, field=u"Другие языки")

        if not other_languages:
            return

        product = get_issue_field(issue, u"Название продукта/запуска")
        subj = u"Мониторинг применения Правил разработки: %s" % product

        if nirvana_check(issue):
            nirvana_audit = True
            t = self.env.get_template("language-notify.tpl")
            body = t.render(
                nirvana_audit=nirvana_audit,
            )
            issue = self.startrek.issues[issue.key]
            issue.comments.create(text=body)
        else:
            issues = self.startrek.issues.find(
            u'Queue: %s AND Summary: #"%s"'
            % (self.config.get("language_notify", "queue"), subj))

            if len(issues):
                return

            developers = ", ".join(self._get_followers(issue))
            repo_urls = ", ".join([r[0] for r in self.get_repo_urls(issue)])
            followers = self._config_getlist("language_notify", "cc")
            t = self.env.get_template("language-notify.tpl")
            body = t.render(
                developers=developers,
                product=product,
                other_languages=other_languages,
                repo=repo_urls,
            )

            notify_issue = self.startrek.issues.create(
                queue=self.config.get("language_notify", "queue"),
                summary=subj,
                assignee=self.config.get("language_notify", "assignee"),
                description=body,
                followers=followers,
            )

    def _get_answers(self, issue):
        answers = {
            "possible_ssrf": False,
            "user_files": False,
            "payment": False,
            "outsourcing": False,
            "promo": False,
            "browser": False,
            "landing": False,
            "too_late": False,
            "admin_dc": False,
            "yql_nv": False,
        }

        if get_issue_field(issue, field=u"Скачивание удаленных ресурсов") == u"Да":
            answers["possible_ssrf"] = True
        if get_issue_field(issue, field=u"Загрузка файлов") == u"Да":
            answers["user_files"] = True
        if (
            get_issue_field(issue, field=u"В приложении есть платежные механики?")
            == u"Да"
        ):
            answers["payment"] = True
        if get_issue_field(issue, field=u"Промо") == u"Да":
            answers["promo"] = True
        if get_issue_field(issue, field=u"Лендинг на Конструкторе (LPC)") == u"Да":
            answers["landing"] = True
        if get_issue_field(issue, field=u"Лендинг на Конструкторе (OSP)") == u"Да":
            answers["landing"] = True
        if get_issue_field(issue, field=u"Лендинг внешнего конструктора") == u"Да":
            answers["landing"] = True
        if get_issue_field(issue, field=u"Аутсорсинг") == u"Да":
            answers["outsourcing"] = True
        if get_issue_field(issue, field=u"Сборка браузера") == u"Да":
            answers["browser"] = True
        if get_issue_field(issue, field=u"Админка") == u"Да":
            answers["admin_dc"] = True
        if get_issue_field(issue, field=u"YQL и базы данных") == u"Да":
            answers["yql_nv"] = True
        issue = self.startrek.issues[issue.key]
        if issue.deadline:
            deadline_datetime = datetime.datetime.strptime(issue.deadline, "%Y-%m-%d")
            # launches happen during working hours
            deadline_datetime += datetime.timedelta(hours=10)
            delta = deadline_datetime - datetime.datetime.now()
            if delta.days < 14:
                answers["too_late"] = True
                # we need to mark this kind of issues
                issue.tags.append("service_late")
                issue.update(tags=issue.tags, ignore_version_change=True)

        if not issue.productionURL:
            if nirvana_check(issue):
                pass
            else:
                answers["no_production_url"] = True
        if not self._get_testing_urls(issue):
            if nirvana_check(issue):
                pass
            else:
                answers["no_testing_url"] = True
        return answers

    def _calculate_risk_level(self, metrics, answers):
        risk_level = 1

        # Answers
        if answers["possible_ssrf"]:
            risk_level += 1
        if answers["user_files"]:
            risk_level += 1
        if answers["outsourcing"]:
            risk_level += 1
        if answers["payment"]:
            risk_level += 1
        if answers["promo"]:
            risk_level -= 1
        if answers["landing"]:
            risk_level -= 1
        if answers["browser"]:
            risk_level += 3

        # Metrics
        # We should review such service!
        if not metrics:
            risk_level += 5
            return risk_level

        if metrics.get("is_critical", False):
            risk_level += 1
        if not metrics["molly"]["status"]:
            risk_level += 1
        if metrics["molly"].get("last_scan_vulns", 0) > 3:
            risk_level += 1
        if not metrics["secaudit"]["status"]:
            risk_level += 1
        if len(metrics["secaudit"].get("issues", [])) > 3:
            risk_level += 1
        if not metrics["bugbounty"]["status"]:
            risk_level += 1
        # TODO
        if not metrics["sast"]["status"]:
            risk_level += 0
        if not metrics["training"]["status"]:
            risk_level += 0

        return risk_level

    def _is_correct_description(self, issue):
        if not issue.description:
            text = self.env.get_template("empty-description.tpl").render()
            issue = self.startrek.issues[issue.key]
            transition = issue.transitions["close"]
            transition.execute(comment=text, resolution="invalid")
            return False
        return True

    def _is_valid_issue_origin(self, issue, autoclose=True):
        # FIXME TOOLSUP-11074
        if "newform" or "dc_nv_form" in issue.tags:
            return True

        if issue.createdBy.login in self._config_getlist(
            "plugins.secaudit", "allowed_authors"
        ):
            return True
        if autoclose:
            t = self.env.get_template("form-only.tpl")
            body = t.render()
            # FIXME
            issue = self.startrek.issues[issue.key]
            transition = issue.transitions["close"]
            transition.execute(comment=body, resolution="invalid")
        return False

    def _get_issue_st_queue(self, issue):
        st_queue = get_issue_field(issue, field=u"Трекер")
        if not st_queue:
            return None
        st_queue = st_queue.strip().upper()
        # URL
        if "/" in st_queue:
            st_queue = list(filter(lambda x: x, st_queue.split("/")))[-1]
        # Ticket
        st_queue = st_queue.split("-")[0]
        # Trash
        if re.match("^[A-Z0-9]+$", st_queue):
            return st_queue
        return None

    def _auto_approve(self, issue):
        if get_issue_field(issue, field="Лендинг на Конструкторе (LPC)").startswith(u"Да") or \
            get_issue_field(issue, field='Лендинг на Конструкторе (OSP)').startswith(u'Да') or \
            get_issue_field(issue, field='Лендинг внешнего конструктора').startswith(u'Да'):
            issue = self.startrek.issues[issue.key]
            if not (self._get_testing_urls(issue) or issue.productionURL):
                t = self.env.get_template("provide-landing-url.tpl")
                body = t.render()
                # FIXME
                transition = issue.transitions["close"]
                transition.execute(comment=body, resolution="invalid")
                return
            issue.tags.append("autoroll")
            issue.update(tags=issue.tags, ignore_version_change=True)

    def _auto_assign(self, issue, first_line=False, abc_id=0):
        if issue.assignee:
            return True

        followers = issue.followers
        components = issue.components

        mentors = {}
        logins = []
        default_assignee = self.config.get("plugins.secaudit", "default_assignee")

        author = issue.createdBy.login
        boss_list = self.staff.get_person_chief_list(author)
        sersec_bosses = [boss for boss in boss_list if boss in self.sersec]
        if get_issue_field(issue, field=u"Сборка браузера"):
            logins = self._config_getlist("plugins.secaudit", "browser_sib_assignee")
            # Assign browser task immediately
            first_line = False
            bro_follower = self.config.get("plugins.secaudit", "bro_follower")
            followers.append(self.startrek.users.get(bro_follower))
        elif abc_id in self._config_getintlist("plugins.secaudit", "pcidss_services"):
            logins = self._config_getlist("plugins.secaudit", "pcidss_assignee")
        elif sersec_bosses:
            boss = sersec_bosses.pop()
            logins = self.sersec[boss].keys()
            components.append(list(self.sersec[boss].values())[0])
        else:
            logins = self._config_getlist("plugins.secaudit", "audit_assignee")

        logins = [login for login in logins if self.gap.get_working_today(login)]

        if not logins or first_line:
            logins = [
                default_assignee,
            ]

        r = random.SystemRandom()
        assignee = r.choice(logins)

        # FIXME
        issue = self.startrek.issues[issue.key]
        if mentors.get(assignee):
            followers.append(mentors.get(assignee))
        return issue.update(
            assignee=assignee,
            ignore_version_change=True,
            followers=followers,
            components=components,
        )

    def get_unchecked_issues(self):
        st_query = u"Queue: SECAUDIT " + u"Resolution: empty() " + u"Tags:!processed"
        issues = self.startrek.issues.find(st_query)
        return issues

    def get_issues_for_assign(self):
        st_query = (
            u"Queue: SECAUDIT "
            + u"Resolution: empty() "
            + u"Assignee: empty() "
            + u"Tags: processed"
        )
        issues = self.startrek.issues.find(st_query)
        return issues

    def _set_checked(self, issue, metrics={}, answers={}, risk_level=0, errors=[]):

        issue = self.startrek.issues[issue.key]
        issue.tags.append("processed")
        if errors:
            issue.tags.append("errors")

        t = self.env.get_template("report.tpl")
        body = t.render(
            issue=issue,
            metrics=metrics,
            answers=answers,
            risk_level=risk_level,
            errors=errors,
        )
        # FIXME
        issue.update(tags=issue.tags, ignore_version_change=True)
        items = [a for a in answers.items() if a[0] not in ("promo", "landing")]
        if errors or any(map(lambda r: r[1], items)):
            # FIXME
            issue = self.startrek.issues[issue.key]
            issue.comments.create(text=body)

    def _init_infra_audit(self, issue):
        targets = get_issue_field(issue, u"Список серверов инфраструктуры сервиса")
        deadline = get_issue_field(issue, u"Сроки")

        if not targets:
            return

        infra_component = self.config.getint("plugins.secaudit", "infra_component")
        infra_assignee = self.config.get("plugins.secaudit", "infra_assignee")
        t = self.env.get_template("infra.tpl")
        body = t.render(targets=targets, deadline=deadline)
        subj = issue.summary + u" [Инфраструктура]"

        issues = self.startrek.issues.find(u'Queue: SECAUDIT AND Summary: #"%s"' % subj)

        if len(issues):
            return

        infra_issue = self.startrek.issues.create(
            queue="SECAUDIT",
            summary=subj,
            assignee=infra_assignee,
            description=body,
            components=[
                infra_component,
            ],
        )
        link = issue.links.create(issue=infra_issue.key, relationship="relates")

    def _force_molly_control(self, issue, st_queue):
        if not (get_issue_field(issue, field=u"Молли") == u"Нет"):
            return

        subj = u"Подключить сканер уязвимостей Молли"
        t = self.env.get_template("molly.tpl")
        body = t.render()
        followers = self._config_getlist("plugins.secaudit", "molly_cc")

        issues = self.startrek.issues.find(
            u'Queue: %s AND Summary: #"%s"' % (st_queue, subj)
        )

        if len(issues):
            return

        new_issue = self.startrek.issues.create(
            queue=st_queue,
            summary=subj,
            assignee=issue.createdBy.login,
            description=body,
            followers=followers,
        )
        link = issue.links.create(issue=new_issue.key, relationship="relates")

    def _force_csp(self, issue, st_queue):
        if get_issue_field(issue, field=u"CSP") == u"Да":
            return

        csp_related = [u"Веб-приложение", u"Промо", u"Расширение для браузера"]
        tmp = list(
            filter(lambda x: get_issue_field(issue, field=x) == u"Да", csp_related)
        )

        if len(tmp) == 0:
            return

        subj = u"Внедрить Content Security Policy"
        t = self.env.get_template("csp.tpl")
        body = t.render()
        followers = self._config_getlist("plugins.secaudit", "csp_cc")
        issues = self.startrek.issues.find(
            u'Queue: %s AND Summary: #"%s"' % (st_queue, subj)
        )

        if len(issues):
            return

        csp_issue = self.startrek.issues.create(
            queue=st_queue,
            summary=subj,
            assignee=issue.createdBy.login,
            description=body,
            followers=followers,
            priority="critical",
        )
        link = issue.links.create(issue=csp_issue.key, relationship="relates")

    def _force_tvm(self, issue, st_queue):
        if get_issue_field(issue, field=u"TVM") == u"Да":
            return

        if nirvana_check(issue):
            # TODO: add tpl for tvm mech in NV
            pass
        else:
            subj = u"Внедрить механизм межсервисной аутентификации TVM"
            t = self.env.get_template("tvm.tpl")
            body = t.render()

            author = issue.createdBy.login
            boss_list = self.staff.get_person_chief_list(author)
            boss = None
            intersection = [boss for boss in boss_list if boss in self.sersec.keys()]
            if intersection:
                boss = intersection.pop()

            followers = self._config_getlist("plugins.secaudit", "tvm_cc")
            if boss in self.sersec:
                followers = list(self.sersec[boss].keys())

            issues = self.startrek.issues.find(
                u'Queue: %s AND Summary: #"%s"' % (st_queue, subj)
            )

            if len(issues):
                return

            tvm_issue = self.startrek.issues.create(
                queue=st_queue,
                summary=subj,
                assignee=issue.createdBy.login,
                description=body,
                followers=followers,
            )
            link = issue.links.create(issue=tvm_issue.key, relationship="relates")

            # reload issue to avoid st exceptions
            tvm_issue = self.startrek.issues[tvm_issue.key]
            tvm_issue.update(
                tags=tvm_issue.tags + ["tvm2", "sec_form"], ignore_version_change=True
            )

    def _summon_antifraud(self, issue):
        if get_issue_field(
            issue, field=u"В приложении есть платежные механики?"
        ).startswith(u"Да"):
            t = self.env.get_template("antifraud.tpl")
            body = t.render()
            issue = self.startrek.issues[issue.key]
            issue.comments.create(text=body, summonees=[self.antifraud_team])

    def _force_pinning(self, issue, st_queue):
        if get_issue_field(issue, field=u"Certificate Pinning") == u"Да":
            return
        subj = u'Внедрить "пиннинг" сертификатов в приложении'
        t = self.env.get_template("pinning.tpl")
        body = t.render()
        followers = self._config_getlist("plugins.secaudit", "pinning_cc")

        issues = self.startrek.issues.find(
            u'Queue: %s AND Summary: #"%s"' % (st_queue, subj)
        )

        if len(issues):
            return

        new_issue = self.startrek.issues.create(
            queue=st_queue,
            summary=subj,
            assignee=issue.createdBy.login,
            description=body,
            followers=followers,
        )
        link = issue.links.create(issue=new_issue.key, relationship="relates")

    def normalize_issue(self, issue):
        # Check author of ticket
        author = get_issue_field(issue, u"Заявку оставил", u"([a-z0-9\-]{2,})")
        if author and author != issue.createdBy.login:
            # FIXME
            issue = self.startrek.issues[issue.key]
            issue.update(createdBy=author, ignore_version_change=True)
        # Set deadline
        deadline = get_issue_field(issue, u"Сроки")
        if deadline and deadline != issue.deadline:
            # FIXME
            issue = self.startrek.issues[issue.key]
            issue.update(deadline=deadline, ignore_version_change=True)
        # Add followers
        if not issue.followers:
            followers = self._get_followers(issue)
            if followers:
                # FIXME
                issue = self.startrek.issues[issue.key]
                issue.update(followers=followers, ignore_version_change=True)
        # Set Production URL
        if not issue.productionURL and not nirvana_check(issue):
            prod_url = get_issue_field(issue, u"URL прода")
            # FIXME add validation for Yandex domains instead of testing
            if prod_url and self._is_valid_testing_uri(prod_url):
                up = urllib.parse.urlparse(prod_url)
                new_up = list(up[:])
                new_up[0] = "https"
                if up.netloc == "yandex.ru":
                    new_path = up.path
                    if up.path.count("/") > 1:
                        new_path = "/".join(up.path.split("/")[:3])
                    new_up[2] = new_path
                else:
                    new_up[2] = "/"
                new_up[3] = ""  # we don't
                new_up[4] = ""  # need
                new_up[5] = ""  # query string
                norm_prod_url = urllib.parse.urlunparse(new_up)

                issue = self.startrek.issues[issue.key]
                issue.update(productionURL=norm_prod_url, ignore_version_change=True)

        if get_issue_field(issue, field=u"Сборка браузера"):
            service_slug = "deskbrowser"
            service = self.abc.get_service_by_slug(service_slug)
            if service:
                issue = self.startrek.issues[issue.key]
                issue.update(abcService=service["id"], ignore_version_change=True)

        # Set ABC service
        if not issue.abcService:
            abc_url = get_issue_field(issue, u"ABC URL")
            service = None
            if abc_url.startswith("https://abc.yandex-team.ru/services/"):
                service_slug = abc_url.strip("/").split("/").pop()
                if service_slug:
                    complex_slug = service_slug.split("?")
                    if complex_slug:
                        service_slug = complex_slug[0]
                    service = self.abc.get_service_by_slug(service_slug)
                if service and service.get("id"):
                    # FIXME
                    issue = self.startrek.issues[issue.key]
                    issue.update(
                        abcService=service.get("id"), ignore_version_change=True
                    )

        # Set component
        if not len(issue.components):
            components = []
            if get_issue_field(issue, field=u"Мобильное приложение").startswith(u"Да"):
                components.append(
                    self.config.getint("plugins.secaudit", "mobile_component")
                )

            if get_issue_field(issue, field=u"Промо") == u"Да":
                components.append(
                    self.config.getint("plugins.secaudit", "promo_component")
                )

            if get_issue_field(issue, field=u"Сборка браузера"):
                components.append(
                    self.config.getint("plugins.secaudit", "browser_component")
                )

            if get_issue_field(
                issue, field=u"Обработка данных пользователей"
            ).startswith(u"Да") and not nirvana_check(issue):
                components.append(
                    self.config.getint("plugins.secaudit", "dpia_component")
                ) 

            if get_issue_field(
                issue, field=u"В приложении есть платежные механики?"
            ).startswith(u"Да"):
                components.append(
                    self.config.getint("plugins.secaudit", "payment_component")
                )
            if components:
                # FIXME
                issue = self.startrek.issues[issue.key]
                issue.update(components=components, ignore_version_change=True)

        # Set Geo components
        self._add_geo_components(issue)
        issue.update(followers=self._get_service_followers(issue), ignore_version_change=True)

        return issue

    def _add_geo_components(self, issue):
        """
        If issue's ABC service have slug starting with 'maps-front' or has maps-core (id=1976) as an ancestor
        add special headers
        """
        if not issue.abcService:
            return
        for service in issue.abcService:
            info = self.abc.get_service_by_id(service.id, fields='ancestors,slug')
            ancestor_slugs = list(map(lambda service: service.get('slug'), info.get('ancestors')))
            slug = info.get('slug')
            if slug.startswith('maps-front'):
                issue.components.append(106663)  # maps-front component id
            if any(map(lambda slug: slug == 'maps-core', ancestor_slugs)):
                issue.components.append(106664)  # maps-core component id
        issue.update(components=issue.components, ignore_version_change=True)

    def _get_service_followers(self, issue):
        """
        If issue is created by some service member, add some people to followers, see core.common.boss_followers_mapping
        """
        followers = []
        for boss in self.staff.get_person_chief_list(issue.createdBy.login):
            if boss in boss_followers_mapping:
                followers += boss_followers_mapping[boss]
        return followers
    
    def _get_component_by_id(self, components, component_id):
        for c in components:
            if c.id == component_id:
                return c
        return None

    def _config_getlist(self, section, option):
        return [i.strip() for i in self.config.get(section, option).split(",")]

    def _get_followers(self, issue):
        result = []
        tmp = get_issue_field(issue, u"Разработчики")
        if nirvana_check(issue):
            tmp_2 = get_issue_field(issue, u"Администратор домена")
            tmp = tmp + ', ' + tmp_2
        separator = " "
        if "," in tmp:
            separator = ","
        followers = [f.strip().lower() for f in tmp.split(separator)]
        for f in followers:
            if "@" in f:
                i = f.split("@")[0]
            else:
                i = f
            if re.findall(u"[a-z0-9\-]{2,}", i):
                result.append(i)

        return result

    def _get_testing_urls(self, issue):
        res = []
        regex = u"URL тестинга:\s+(?:[a-zA-Z]|[0-9]|[$-_@.&+]|[!*\(\),]|"
        regex += u"(?:%[0-9a-fA-F][0-9a-fA-F]))+"
        URL_RE = re.compile(regex)
        for item in URL_RE.findall(issue.description):
            item = re.sub(u"URL тестинга:\s+", "", item)
            if item[:4] != "http" and item != "n/a":
                res.append("https://" + item)
                res.append("http://" + item)
            else:
                res.append(item)
        res = list(filter(Secaudit._is_valid_testing_uri, res))
        return res

    def _get_apk_url(self, issue):
        res = []
        regex = u"APK URL:\s+(?:[a-zA-Z]|[0-9]|[$-_@.&+]|[!*\(\),]|"
        regex += u"(?:%[0-9a-fA-F][0-9a-fA-F]))+"
        URL_RE = re.compile(regex)
        for item in URL_RE.findall(issue.description):
            item = re.sub(u"APK URL:\s+", "", item)
            if item[:4] != "http" and item != "n/a":
                res.append("https://" + item)
                res.append("http://" + item)
            else:
                res.append(item)
        res = list(filter(Secaudit._is_valid_testing_uri, res))
        return res

    @staticmethod
    def _is_valid_beta_url(uri):
        if not uri or uri == "n/a":
            return False
        if uri == "https://None" or uri == "http://None":
            return False
        uri = uri.strip()
        if uri.lower().startswith("https://beta.m.soft.yandex.ru/"):
            return True
        return False

    @staticmethod
    def _is_valid_testing_uri(uri):
        if not uri:
            return False
        if uri == "https://URL" or uri == "http://URL":
            return False
        if uri == "https://None" or uri == "http://None":
            return False
        if uri == "n/a":
            return False
        if uri.find("https://github.yandex-team.ru") > -1:
            return False
        if uri.find("https://st.yandex-team.ru") > -1:
            return False
        if uri.find("https://lp-constructor.yandex-team.ru") > -1:
            return False
        return True

    # https://github.yandex-team.ru/music/remusic/tree/master/api/v2.1 ->
    # https://github.yandex-team.ru/music/remusic.git and master
    @staticmethod
    def _crack_github_url_tree(url):
        if "/tree/" in url:
            sp = url.split("/tree/")
            return sp[0] + ".git", sp[1].split("/")[0]
        return url, "master"

    @staticmethod
    def _crack_github_url_sharp(url):
        if "#" in url:
            sp = url.split("#")
            repo_url = sp[0] if sp[0].endswith(".git") else sp[0] + ".git"
            branch = sp[1] if sp[1] else "master"
            return repo_url, branch
        return url, "master"

    @staticmethod
    def _add_dot_git(url):
        if url.endswith("/"):
            url = url[:-1]
        return url if url.endswith(".git") else url + ".git"

    @staticmethod
    def get_repo_urls(issue):
        repos = []
        INTERNAL_REPOS = {
            "github": {
                "pat": [re.compile("(\w+)://github\.yandex-team\.ru/(.*?)([)\s\n\r])")],
                "host": "https://github.yandex-team.ru",
            },
            "bitbucket": {
                "pat": [re.compile("(\w+)://bb\.yandex-team\.ru/(.*?)([)\s\n\r])")],
                "host": "https://bb.yandex-team.ru",
            },
            "external_github": {
                "pat": [re.compile("(\w+)://github\.com/(.*?)([)\s\n\r])")],
                "host": "https://github.com",
            },
            "browser_bitbucket": {
                "pat": [
                    re.compile(
                        "(\w+)://bitbucket\.browser\.yandex-team\.ru/(.*?)([)\s\n\r])"
                    )
                ],
                "host": "https://bitbucket.browser.yandex-team.ru",
            },
            "arcadia": {
                "pat": [re.compile("(?:\w+://)a\.yandex\-team\.ru/[\w\-\/\.]+")],
                "host": "svn+ssh://a.yandex-team.ru",
            },
            "sdc_bitbucket": {
                "pat": [
                    re.compile(
                        "(\w+)://bitbucket\.sdc\.yandex-team\.ru/(.*?)([)\s\n\r])"
                    )
                ],
                "host": "https://bitbucket.sdc.yandex-team.ru",
            },
        }

        for k in INTERNAL_REPOS.keys():
            for pat in INTERNAL_REPOS[k]["pat"]:
                for res in pat.findall(issue.description):
                    repo_url, add_path, branch = "", "", ""
                    repo = res[1]
                    if k in {"github", "external_github"}:
                        parts = repo.split("/")
                        if len(parts) < 2:
                            continue
                        repo_url = "/".join(
                            [INTERNAL_REPOS[k]["host"], parts[0], parts[1]]
                        )
                        if len(parts) >= 4 and parts[2] == "tree":
                            branch = parts[3]
                        if len(parts) > 4:
                            add_path = "/" + "/".join(parts[4:])
                        repos.append(
                            dict(url=repo_url, branch=branch, add_path=add_path)
                        )
                    elif k in {"bitbucket", "browser_bitbucket"}:
                        repo_parts = repo.split("?")
                        repo = repo_parts[0]
                        args = repo_parts[1] if len(repo_parts) > 1 else ""
                        parts = repo.split("/")
                        if len(parts) < 2:
                            # skip incorrect url
                            continue
                        elif len(parts) == 2:
                            # url: https://bb.yandex-team.ru/PRJ/REPO
                            repo_url = "/".join(
                                [INTERNAL_REPOS[k]["host"], "scm", parts[0], parts[1]]
                            )
                        elif len(parts) >= 3 and parts[0] == "scm":
                            # url: https://bb.yandex-team.ru/scm/PRJ/REPO
                            repo_url = "/".join(
                                [INTERNAL_REPOS[k]["host"], "scm", parts[1], parts[2]]
                            )
                        elif (
                            len(parts) >= 4
                            and (parts[0] == "projects" or parts[0] == "users")
                            and (parts[2] == "repos")
                        ):
                            # url: https://bb.yandex-team.ru/projects/PRJ/repos/REPO/browse/PATH/IN/REPO
                            repo_url = "/".join(
                                [INTERNAL_REPOS[k]["host"], "scm", parts[1], parts[3]]
                            )
                            if len(parts) > 5:
                                add_path = "/" + "/".join(parts[5:])
                            branch = args.split("%2F")[-1]
                        else:
                            # skip incorrect url
                            continue
                        repos.append(
                            dict(url=repo_url, branch=branch, add_path=add_path)
                        )
                    elif k == "arcadia":
                        repo_url = res[:-1] if res[-1] == "/" else res
                        repo_parts = repo_url.split("/")
                        if "." in repo_parts[-1]:
                            repo_url = "/".join(repo_parts[:-1])
                        repos.append(dict(url=repo_url, branch="", add_path="/"))
                    else:
                        repo_url = res[:-1] if res[-1] == "/" else res  # todo: Improve
                        repos.append(dict(url=repo_url, branch="", add_path=""))
        return repos


def get_issue_field(issue, field, pattern="(.+)"):
    m = re.findall(field + ":\s+" + pattern, issue.description)
    if not m:
        return ""
    value = m[0].strip()
    if value == "None":
        return ""
    return value

def nirvana_check(issue):
    result = "Nirvana" in [c.name for c in issue.components]
    return result
