# -*- coding: utf-8 -*-
from security.c3po.components.core.plugins import BasePlugin
from startrek_client import Startrek
from security.yaseclib.abc import Abc
from security.yaseclib.idm import Idm
from security.yaseclib.staff import Staff


class SecauditImpulseIdm(BasePlugin):
    title = "imPulse SECAUDIT project ACL"
    desc = (
        "Adding View role in SECAUDIT-XXX project "
        "for linked ABC-service and ticket followers"
    )

    def setup(self):
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )
        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )
        self.idm = Idm(
            base_url=self.config.get("idm", "url"),
            token=self.config.get("idm", "token"),
        )

    def main(self):
        self._walk_secaudit()

    def _walk_secaudit(self):
        st_query = (
            "Queue: SECAUDIT "
            "Tags: processed "
            "Tags: !impulse_idm_processed "
            "Tags: !impulse_idm_error "
            "Resolution: empty() "
            "Created: < now()"
        )

        issues = self.startrek.issues.find(st_query)

        for issue in issues:
            self._process_secaudit(issue)

    def _process_error(self, issue):
        tags = issue.tags
        if "impulse_idm_error_1" in tags:
            tags.remove("impulse_idm_error_1")
            tags.append("impulse_idm_error_2")
        elif "impulse_idm_error_2" in tags:
            tags.remove("impulse_idm_error_2")
            tags.append("impulse_idm_error_3")
        elif "impulse_idm_error_3" in tags:
            tags.remove("impulse_idm_error_3")
            tags.append("impulse_idm_error")
        else:
            tags.append("impulse_idm_error_1")

        issue.update(
            tags=tags,
            ignore_version_change=True,
            params={"notify": "False"},
        )

    def _process_secaudit(self, issue):
        idm_role_path = "/secaudit/{}/view/".format(issue.key)
        for service in issue.abcService:
            service_id = service.as_dict()["id"]
            service_info = self.abc.get_service_by_id(service_id)
            service_slug = service_info.get("slug")
            if service_slug is None:
                continue
            for role in ["development", "services_management"]:
                entity_slug = "svc_{}_{}".format(service_slug, role)
                group_info = self.staff.get_group_info_by_url(entity_slug, ["id"])
                if group_info and "id" in group_info:
                    group_id = group_info["id"]
                else:
                    continue
                res = self.idm.request_role(
                    system="impulse", group=group_id, path=idm_role_path
                )
                if not res or ("error_code" in res and res["error_code"] != "CONFLICT"):
                    self._process_error(issue)
                    return

        followers = [follower.login for follower in issue.followers]
        followers.append(issue.createdBy.login)
        for follower in set(followers):
            if follower.startswith(("robot-", "zomb-")):
                continue
            res = self.idm.request_role(
                system="impulse", user=follower, path=idm_role_path
            )
            if not res or ("error_code" in res and res["error_code"] != "CONFLICT"):
                self._process_error(issue)
                return
        issue.tags.append("impulse_idm_processed")
        issue.update(
            tags=issue.tags,
            ignore_version_change=True,
            params={"notify": "False"},
        )
