import datetime
import logging

import startrek_client
from security.c3po.components.core.common import service_mapping
from security.c3po.components.core.plugins import BasePlugin
from security.c3po.components.core.util import get_duty_tag, get_config
from security.yaseclib.staff import Staff
from startrek_client import Startrek


class SecdutyLinker(BasePlugin):
    """docstring for SecdutyLinker"""

    title = "Linker for issues resolved during duty"
    desc = (
        "Находим тикеты, в которых исполнитель — "
        "текущий дежурный по БЮ и привязываем к тикету на дежурство из SECDUTY"
    )

    def setup(self):
        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

    def main(self):
        today = datetime.date.today()
        duty_query = (
            "(Status:!Closed "
            "Queue:SECDUTY "
            "Deadline:!empty()) "
            " OR "
            "(Queue: CLOUDDUTY "
            "Components: 47232 "
            "Status:!closed "
            "Assignee:!empty())"
        )
        duty_issues = self.startrek.issues.find(duty_query)

        # there's no easy way of attributing tasks in these queues
        complex_queues = ["BOUNTYREPORT", "IDMREQ"]

        for duty_issue in duty_issues:
            duty_start_date = datetime.datetime.strptime(
                duty_issue.createdAt[:10], "%Y-%m-%d"
            ).date()
            days_diff = (today - duty_start_date).days

            st_query = (
                "((Queue:DOSTUPREQ,CLOUD,APPSECDUTY,SECTASK,CLOUDSECURITY,BOUNTYREPORT,SECREVIEW "
                "Assignee:%s@) "
                "OR "
                "(Queue: IDMREQ "
                "Followers:%s@)) "
                "Relates:!%s "
                "Created: >=today()-%sd "
                "Created: <=today() "
                % (
                    duty_issue.assignee.login,
                    duty_issue.assignee.login,
                    duty_issue.key,
                    days_diff,
                )
            )
            issues = self.startrek.issues.find(st_query)

            duty_tag = get_duty_tag(duty_issue)
            if not duty_tag:
                logging.warning(f'No duty tag {duty_issue.key}')
                continue

            for issue in issues:
                try:
                    creator = issue.createdBy.login
                    if self.staff.is_robot(creator):
                        creator = self.staff.get_robot_responsible(creator)
                    boss_list = self.staff.get_person_chief_list(creator)
                    svc_check = [
                        boss
                        for boss in boss_list
                        if boss in service_mapping[duty_tag]
                    ]
                    if svc_check or issue.queue.key in complex_queues:
                        issue.links.create(
                            issue=duty_issue.key,
                            relationship="relates",
                            params={"notify": "False"},
                        )
                except startrek_client.exceptions.UnprocessableEntity:
                    # this means issue is already linked
                    pass

    def get_shifts_cloud(self):
        query = (
            "Queue: CLOUDDUTY "
            "Components: 47232 "
            "Status:!closed "
            '"Sort By": Created ASC'
        )
        issues = self.startrek.issues.find(query)
        result = []

        for issue in issues:
            if issue.assignee:
                result.append(issue.assignee.login)

        return result


if __name__ == '__main__':
    config = get_config('/home/ya-andrei/Documents/c3po/generic_config.ini')
    sl = SecdutyLinker(config, None)
    sl.setup()
    sl.main()
