# -*- coding: utf-8 -*-
import random

from security.c3po.components.core.plugins import BasePlugin
from security.c3po.components.core.common import secreview_mapping
from startrek_client import Startrek
from security.yaseclib.abc import Abc
from security.yaseclib.gap import Gap
from security.yaseclib.tvm import TVM


class Secreview(BasePlugin):
    title = u"SecReview"
    desc = u"Monitoring of forgotten issues in SECREVIEW queue"

    def setup(self):
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(
            Gap.CALENDAR_YATEAM_TVM_ID
        )

        self.gap = Gap(
            base_url=self.config.get("gap", "url"),
            token=self.config.get("gap", "token"),
            base_calendar_url=self.config.get("calendar", "url"),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket,
        )
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )  # Forming shifts
        self.security_abc_id = self.config.get("plugins.secreview", "security_abc_id")
        self.duty_assignees = {}
        shifts = self.abc.get_shifts(self.security_abc_id)
        for shift in shifts:
            shifts_slug = shift["schedule"]["slug"]
            duty_assignee = shift["person"]["login"]
            self.duty_assignees[shifts_slug] = duty_assignee

        self.service_mapping = secreview_mapping

        if "coresec_everyday" in self.duty_assignees:
            self.service_mapping.update(
                {
                    "87554": [
                        self.duty_assignees["coresec_everyday"]
                    ],  # Search and related
                    "87556": [self.duty_assignees["coresec_everyday"]],  # Toloka
                }
            )

    def main(self):
        self._walk_secreview()
        self.tvm.stop()

    def _walk_secreview(self):
        st_query = (
            "Queue: SECREVIEW "
            "Tags: !processed "
            "Resolution: empty() "
            "Created: < now()"
        )

        issues = self.startrek.issues.find(st_query)

        for issue in issues:
            self._process_secreview(issue)

    def _process_secreview(self, issue):
        assignee = None

        if issue.components:
            candidates = [
                officer
                for officer in self.service_mapping[issue.components[0].id]
                if self.gap.get_working_today(officer)
            ]
            if candidates:
                # feat(dbeltukov): do not reassign to random candidate if component is linked to duty,
                # i.e officer is already assigned by lead
                # test on Bank first
                if "104195" in [component.id for component in issue.components]:
                    assignee = issue.assignee or random.choice(candidates)
                else:
                    assignee = random.choice(candidates)
        if not issue.components:
            issue = self.startrek.issues[issue.key]
            transition = issue.transitions["close"]
            transition.execute(
                comment="Заявки на проведения ревью безопасности принимаются ((https://wiki.yandex-team.ru/secreview только через форму)). Пожалуйста, заполните её и мы возьмём заявку в работу.",
                resolution="invalid",
            )
            return False

        if assignee:
            issue.tags.append("processed")

            issue.update(assignee=assignee, tags=issue.tags, ignore_version_change=True)
