# -*- coding: utf-8 -*-
from security.c3po.components.core.plugins import BasePlugin
from jinja2 import Environment, FileSystemLoader
from startrek_client import Startrek


class SecreviewMarket(BasePlugin):
    title = u"Market Security Design Review"
    desc = u"Monitoring Market project queues for issues ready for development"

    def setup(self):
        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "templates"
            )
        )

        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )
        self.project_queues = self._config_getlist(
            "plugins.secreview_market", "project_queues"
        )

    def main(self):
        self._walk_secreview()

    def _walk_secreview(self):
        st_query = (
            "Queue: {} "
            "Tags: !processed "
            "Tags: market_secdesignreview "
            "Resolution: empty() "
            "Status: 24 "
            "Created: < now()".format(",".join(self.project_queues))
        )

        issues = self.startrek.issues.find(st_query)

        for issue in issues:
            self._process_secreview(issue)

    def _process_secreview(self, project_issue):
        comments = project_issue.comments.get_all()
        parsed_comments = []
        for comment in comments:
            parsed_comment = {
                "display_name": "{} {}".format(
                    comment.createdBy.firstName, comment.createdBy.lastName
                ),
                "link": "https://st.yandex-team.ru/{key}#{id}".format(
                    key=project_issue.key, id=comment.longId
                ),
                "created_at": comment.createdAt,
                "text": comment.text,
            }
            parsed_comments.append(parsed_comment)

        followers = [project_issue.createdBy.login]
        if project_issue.assignee and project_issue.assignee.login not in followers:
            followers.append(project_issue.assignee.login)

        t = self.env.get_template("secreview.tpl")
        description = t.render(
            description=project_issue.description,
            comments=parsed_comments,
        )

        created_issue = self.startrek.issues.create(
            queue="SECREVIEW",
            summary="[SecReview] {}".format(project_issue.summary),
            description=description,
            components=[57685],
            followers=followers,
        )
        created_issue.links.create(issue=project_issue.key, relationship="relates")

        project_issue.tags.append("processed")
        project_issue.update(
            tags=project_issue.tags,
            ignore_version_change=True,
        )
