# -*- coding: utf-8 -*-
import logging
import re
from collections import defaultdict

from security.c3po.components.core.common import sectask_mapping, boss_followers_mapping, ABC_ID
from security.c3po.components.core.plugins import BasePlugin
from security.c3po.components.core.util import get_issue_author, safe_func
from security.yaseclib.abc import Abc
from security.yaseclib.gap import Gap
from security.yaseclib.staff import Staff
from security.yaseclib.tvm import TVM
from startrek_client import Startrek


class Sectask(BasePlugin):
    title = u"SecTask"
    desc = u"Monitoring of forgotten issues in SECTASK/security"

    def setup(self):
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(
            Gap.CALENDAR_YATEAM_TVM_ID
        )

        self.gap = Gap(
            base_url=self.config.get("gap", "url"),
            token=self.config.get("gap", "token"),
            base_calendar_url=self.config.get("calendar", "url"),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket,
        )
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )
        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.officers = self._config_getlist("plugins.sectask", "officers")
        self.default_assignee = self.config.get("plugins.sectask", "default_assignee")
        self.abuse_assignee = self.config.get("plugins.sectask", "abuse_assignee")
        self.pastebin_assignee = self.config.get("plugins.sectask", "pastebin_assignee")
        self.pki_code = self.config.getint("plugins.sectask", "pki_component")

        self.service_mapping = sectask_mapping
        self.duty_assignees = {}

        # Forming shifts
        self.security_abc_id = self.config.get("plugins.sectask", "security_abc_id")
        shifts = self.abc.get_shifts(self.security_abc_id)
        self.coresec_assignee = None
        for shift in shifts:
            shifts_slug = shift["schedule"]["slug"]
            duty_assignee = shift["person"]["login"]
            # if shift['replaces']:
            #     duty_assignee = shift['replaces']['person']['login']
            if shifts_slug == "coresec_everyday":
                self.coresec_assignee = duty_assignee
            if self.duty_assignees.get(duty_assignee) is not None:
                self.duty_assignees[duty_assignee].update(
                    self.service_mapping.get(shifts_slug)
                )
            else:
                self.duty_assignees[duty_assignee] = self.service_mapping.get(
                    shifts_slug
                )
        if self.coresec_assignee is None:
            self.coresec_assignee = 'horus'  # Emergency patch. TODO: remove

        # Add Cloud shifts
        shifts_cloud = self.get_shifts_cloud()
        if shifts_cloud:
            cloud_assignee = shifts_cloud[0]
        else:
            cloud_assignee = "axlodin"
        cloud_bosses = {"abash": [64074], "polievkt": [64074]}
        if self.duty_assignees.get(cloud_assignee) is not None:
            self.duty_assignees[cloud_assignee].update(cloud_bosses)
        else:
            self.duty_assignees[cloud_assignee] = cloud_bosses

        # Re-mapping to {boss: assignee}
        self.bosses = defaultdict(dict)
        for key, values in self.duty_assignees.items():
            if not values:
                logging.error(f'Check mappings for {key}!\nduty_assignees={self.bosses}')
                continue
            for value in values.keys():
                self.bosses[value][key] = values[value]

    def main(self):
        self._walk_sectask()
        self.tvm.stop()

    def _walk_sectask(self):
        st_query = (
            "Queue: SECTASK "
            "Resolution: empty() "
            "Components: !%d "
            "Assignee: empty() "
        )
        st_query = st_query % self.pki_code
        issues = self.startrek.issues.find(st_query)
        for issue in issues:
            logging.info(f'Processing issue {issue.key}')
            self._process_sectask(issue)

    @safe_func
    def _process_sectask(self, issue):

        components = issue.components
        followers = []
        if self._check_mvd(issue):
            comment = (
                u"Запрос не соответствует установленной процедуре. " u"Тикет закрываю."
            )
            self._close_issue(issue, comment)
            return
        assignee = None

        if self._check_edu(issue):
            assignee = "a-maksimova"

        if self._check_abuse(issue):
            assignee = self.abuse_assignee

        if self._check_pastebin(issue):
            assignee = self.pastebin_assignee

        if Sectask._check_software_analysis(issue):
            followers = self.abc.get_people_by_id(
                ABC_ID.ACCESS_GRANTEE,
                with_descendants=False,
                use_inheritance_settings=True,
                role_scope='analitics'
            )
            assignee = 'katsello'

        if not assignee:
            assignee, component = self._choose_assignee_n_component(issue)
            if component:
                components.extend(component)
            if self._check_tracker_summon(issue) and assignee:
                for linked in issue.links.get_all():
                    if linked.object.key in issue.summary:
                        linked_issue = self.startrek.issues[linked.object.key]
                        linked_issue.followers.append(assignee)
                        linked_issue.update(
                            ignore_version_change=True, followers=linked_issue.followers
                        )

        issue.tags.append("processed")
        followers += self._get_service_followers(issue)
        if assignee:
            return issue.update(
                assignee=assignee,
                tags=issue.tags,
                components=components,
                followers=followers,
                ignore_version_change=True,
            )

    def _choose_assignee_n_component(self, issue):
        try:
            boss_list = self.staff.get_person_chief_list(get_issue_author(issue))
            bosses_candidates = [boss for boss in boss_list if boss in self.bosses]
        except Exception:
            # Sometimes issue author is a service or external person
            bosses_candidates = []
        volunteer = self._find_volunteer(issue)
        candidates = []
        component = [74524]

        # Core Team can help with a strange service
        assignee = self.coresec_assignee

        if bosses_candidates:
            boss = bosses_candidates.pop()
            component = list(self.bosses[boss].values())[0]
            if volunteer:
                candidates.extend(volunteer)
            else:
                candidates = list(self.bosses[boss].keys())
            if candidates:
                assignee = candidates[0]

        return assignee, component

    def _find_volunteer(self, issue):
        volunteer = []
        comments = issue.comments.get_all()
        if comments:
            for comment in comments:
                login = comment.createdBy.login
                # Even if somebody commented while he's away,
                # it's more reasonable to assign task to present officers.
                if login in self.officers:
                    if self.gap.get_working_today(login):
                        volunteer.append(login)
                        break
        return volunteer

    def _check_edu(self, issue):
        subjs = [
            "NDA в Яндексе",
            "Внимание! Вы попались на фишинг!",
            "курс NDA",
            "«Антифишинг»: обязательный курс",
            "Информационная безопасность в Яндексe //"
            " Information security at Yandex",
            "Безопасная разработка в Яндексе",
        ]
        issue_subject = issue.summary.lower()
        for subj in subjs:
            if subj.lower() in issue_subject:
                return True
        return False

    def _check_mvd(self, issue):
        description = issue.description
        if description:
            marker = (
                r"\[Created via e\-mail received from: "
                r"[a-zA-Z0-9\.]+@(cyberdept|cybercop|mvd)\.ru\]$"
            )
            matches = re.findall(marker, description)
            if matches:
                return True
        return False

    @staticmethod
    def _check_software_analysis(issue) -> bool:
        for tag in issue.tags:
            if tag == 'saas_assessment':
                return True
        return False

    def _check_abuse(self, issue):
        description = issue.description
        if description:
            marker = (
                r"\[Created via e\-mail received from: "
                r"securityoperations\@markmonitor\.com\]$"
            )
            matches = re.findall(marker, description)
            if matches:
                return True
        return False

    def _get_service_followers(self, issue):
        """
        If issue is created by some service member, add some people to followers, see core.common.boss_followers_mapping
        """
        followers = [f.login for f in issue.followers]
        for boss in self.staff.get_person_chief_list(issue.createdBy.login):
            if boss in boss_followers_mapping:
                followers += boss_followers_mapping[boss]
        return followers

    def _check_tracker_summon(self, issue):
        description = issue.description
        if description:
            marker = (
                r"\[Created via e\-mail received from: "
                r"st\@yandex\-team\.ru\]$"
            )
            matches = re.findall(marker, description)
            links = issue.links.get_all()
            summary_marker = "invites the mailing list to comment"
            verdict = matches and links and (summary_marker in issue.summary)
            return verdict
        return False

    def _check_pastebin(self, issue):
        return issue.summary.startswith("Pastebin.com Alerts Notification")

    def _close_issue(self, issue, comment):
        tags = list(issue.tags)
        tags.append("auto_closed")
        issue.update(tags=tags, ignore_version_change=True)
        issue = self.startrek.issues[issue.key]
        transition = issue.transitions["close"]
        transition.execute(comment=comment, resolution="won'tFix")

    def get_shifts_cloud(self):
        query = (
            "Queue: CLOUDDUTY "
            "Components: 47232 "
            "Status:!closed "
            '"Sort By": Created ASC'
        )
        issues = self.startrek.issues.find(query)
        result = []

        for issue in issues:
            if issue.assignee:
                result.append(issue.assignee.login)

        return result
