# -*- coding: utf-8 -*-
import startrek_client
import logging
import traceback
import datetime

from jinja2 import Environment, FileSystemLoader

from security.yaseclib.tvm import TVM
from security.yaseclib.gap import Gap
from security.yaseclib.staff import Staff

from security.c3po.components.core.plugins import BasePlugin


def continue_on_fail(f):
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except:
            logging.exception("[SectaskTicketReviver]\n%s\n" % traceback.format_exc())

    return wrapper


class SectaskTicketReviver(BasePlugin):
    title = u"SectaskTicketReviver"
    desc = (
        u"Updater of forgotten SECTASK tickets."
        + u" Designed to be launched once a day."
    )

    def setup(self):
        self._load_conf()

        self.summon_count = {}
        self.pki_code = self.config.getint(
            "plugins.sectask_ticket_reviver", "pki_component"
        )
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(
            Gap.CALENDAR_YATEAM_TVM_ID
        )

        self.gap = Gap(
            base_url=self.config.get("gap", "url"),
            token=self.config.get("gap", "token"),
            base_calendar_url=self.config.get("calendar", "url"),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket,
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.startrek = startrek_client.Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "comment_templates"
            )
        )

    def main(self):
        if self.gap.is_workday():
            self.walk_sectask()
        self.tvm.stop()

    def _load_conf(self):
        self.no_remind_tag = self.config.get(
            "plugins.sectask_ticket_reviver", "no_remind"
        )

        self.divinities = self._config_getlist(
            "plugins.sectask_ticket_reviver", "divinities"
        )

    @continue_on_fail
    def walk_sectask(self):
        search_query = (
            "Queue: SECTASK "
            + "Status: !closed "
            + "(Deadline: <= today() "
            + "OR Deadline: empty()) "
            + "Components: !%d "
        )
        search_query = search_query % self.pki_code

        issues = self.startrek.issues.find(search_query)
        for issue in issues:
            if self.no_remind_tag in issue.tags:
                self._no_remind(issue)

            elif self._check_update_date(issue) > 5:
                self._process_sectask(issue)

    @continue_on_fail
    def _process_sectask(self, issue):
        """Return True if ticket has to be escalated."""
        template = "summon_sectask.tpl"
        summonee = "horus"

        assignee = issue.assignee
        assignee_is_ok = (
            assignee
            and self.staff.is_yandexoid(assignee.login)
            and not self.staff.is_external(assignee.login)
        )

        # If there's an assignee and he's still working we summon him
        # Else, summon horus or ybuchnev
        if assignee_is_ok:
            summonee = assignee

        # If people don't care, we'll close the issue
        if self._prepare_closing(issue):
            if "close_soon" in issue.tags:
                comments = issue.comments.get_all()
                if comments:
                    comments = comments._data
                if comments[-1].createdBy.login == self.startrek.myself.login:
                    self._close_issue(issue)
                    return False
                issue.tags.remove("close_soon")
                issue.tags.append("no_remind")
                issue.update(tags=issue.tags, ignore_version_change=True)
            else:
                author = issue.createdBy
                author_is_legit = (
                    assignee
                    and self.staff.is_yandexoid(author.login)
                    and not self.staff.is_external(author.login)
                )
                if author_is_legit:
                    summonee = author
                template = "last_call_sectask.tpl"
                issue.tags.append("close_soon")
                issue.update(tags=issue.tags, ignore_version_change=True)

        if issue.status.key in ["needInfo"]:
            summonee = issue.createdBy.login

        self._summon(issue, summonee, template)

        if summonee == issue.createdBy.login:
            issue = self.startrek.issues[issue.key]
            transits = [transit.id for transit in issue.transitions.get_all()]
            if "need_info" in transits:
                transition = issue.transitions["need_info"]
                transition.execute()

        return False

    def _summon(self, issue, users, template):
        to_summon = []
        if not isinstance(users, list):
            users = [users]
        for user in users:
            if isinstance(user, str) or isinstance(user, bytes):
                user = self.startrek.users.get(user)

            # Only humans are allowed
            if user.login.startswith(("robot-", "zomb-")):
                continue

            # Some people just don't like to be bothered
            if user.login in self.divinities:
                user = self.startrek.users.get("sterh")

            # Load balancing
            if user.login in self.summon_count:
                self.summon_count[user.login] += 1
            else:
                self.summon_count[user.login] = 1

            if self.summon_count[user.login] < 3 and self.gap.get_working_today(
                user.login
            ):
                to_summon.append(user)

        if to_summon:
            t = self.env.get_template(template)
            issue.comments.create(
                text=t.render(summon_security=(to_summon != issue.createdBy.login)),
                summonees=to_summon,
            )
            return True

    def _count_my_comments(self, issue):
        comments = issue.comments.get_all()
        summons = 0
        for comment in comments:
            if comment.createdBy.login == self.startrek.myself.login:
                summons += 1

        return summons

    def _prepare_closing(self, issue):
        comments = issue.comments.get_all()
        if comments:
            comments = comments._data
            comments = comments[-3:]
        summons = 0
        for comment in comments:
            if comment.createdBy.login == self.startrek.myself.login:
                summons += 1

        return True if summons == 3 else False

    def _close_issue(self, issue, template="autoclose_sectask.tpl"):
        issue.tags.append("auto_closed")
        issue.update(tags=issue.tags, ignore_version_change=True)
        # FIXME
        issue = self.startrek.issues[issue.key]
        transition = issue.transitions["close"]
        t = self.env.get_template(template)
        transition.execute(comment=t.render(), resolution="won'tFix")

    def _no_remind(self, issue):
        if self._check_update_date(issue) > 45:
            new_tags = list(issue.tags)
            new_tags.remove(self.no_remind_tag)
            issue.update(tags=new_tags, ignore_version_change=True)

    def _get_followers(self, issue):
        result = []
        if issue.followers:
            for follower in issue.followers:
                result.append(follower.login)
        return result

    def _check_update_date(self, issue):
        today = datetime.datetime.now().date()
        update_date = datetime.datetime.strptime(
            issue.updatedAt[:10], "%Y-%m-%d"
        ).date()
        return (today - update_date).days
