import datetime

from security.c3po.components.core.common import iss_tags_mapping
from security.c3po.components.core.plugins import BasePlugin
from security.c3po.components.core.util import get_iss_tag
from security.yaseclib.staff import Staff
from startrek_client import Startrek


class SecurityAssessor(BasePlugin):
    title = u"SecurityAssessor"
    desc = (
        u"Ходим по Трекеру, собираем открытые тикеты с Security: Yes, "
        u"расставляем правильные теги, дедлайны и Security Severity"
    )

    def setup(self):
        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.nonprocess_queues = self._config_getlist(
            "plugins.security_assessor", "nonprocess_queues"
        )

        self.headers = {
            "Content-Type": "application/json",
            "Authorization": "OAuth %s" % self.config.get("st", "token"),
        }

        self.sec_policy = {
            "blocker": "5",
            "critical": "4",
            "normal": "3",
            "minor": "2",
            "trivial": "2",
            "0": None,
            "1": 90,
            "2": 60,
            "3": 31,
            "4": 7,
            "5": 3,
            "6": 1,
        }

        self.service_mapping = iss_tags_mapping
        self.service_tags = set(self.service_mapping.values())

    def main(self):
        self._walk_st()

    def _walk_st(self):
        search_query = ""

        for queue in self.nonprocess_queues:
            search_query += "Queue: !%s " % queue
        search_query = "((Queue: SECALERTS Type: Ошибка) OR (%s)) " % search_query

        exclude_iss_tags = 'OR'.join(map(lambda tag: f' Tags:!{tag} ', self.service_tags))
        search_query += (
            '(Security: notEmpty() '
            f'("Security Severity": empty() OR {exclude_iss_tags} OR Deadline: empty()) '
            'Status: !closed) '
            "OR "
            "(Tags: cloud_vuln) "
        )

        issues = self.startrek.issues.find(search_query)

        today = datetime.date.today()

        for issue in issues:
            self.process_issue(issue, today)

    def process_issue(self, issue, today):
        tags = list(issue.tags)

        iss_tag = get_iss_tag(issue)
        if iss_tag:
            return  # Treat already processed

        assignee = issue.assignee
        qlead = issue.queue.lead
        assignee_ok = assignee and self._is_yandexoid(assignee.login)
        qlead_ok = qlead and self._is_yandexoid(qlead.login)

        candidate = None
        boss = None

        severity = None
        deadline = None

        if assignee_ok:
            candidate = assignee.login
        elif qlead_ok:
            candidate = qlead.login

        if candidate:
            boss_list = self.staff.get_person_chief_list(candidate)
            chiefs = [boss for boss in boss_list if boss in self.service_mapping]
            if chiefs:
                boss = chiefs.pop()
        tags.append(self.service_mapping.get(boss, "iss_search"))

        if issue.securitySeverity and issue.securitySeverity in self.sec_policy:
            severity = issue.securitySeverity
        else:
            severity = self.sec_policy[issue.priority.key]

        if issue.deadline:
            deadline = issue.deadline
        elif severity != "0":
            if severity in self.sec_policy:
                delta = self.sec_policy[severity]
                if delta:
                    deadline = (today + datetime.timedelta(days=delta)).isoformat()

        issue.update(
            deadline=deadline,
            securitySeverity=severity,
            tags=list(set(tags)),
            security="Yes",
            params={"notify": "False"},
            ignore_version_change=True,
        )

    def _is_yandexoid(self, login):
        return (
            login
            and self.staff.is_yandexoid(login)
            and not self.staff.is_external(login)
        )
