# -*- coding: utf-8 -*-
import re

from security.yaseclib.abc import Abc
from startrek_client import Startrek
from security.c3po.components.core.plugins import BasePlugin


class TaxiSecurity(BasePlugin):
    title = u"TaxiSecurity"
    desc = u"Auto-assignment for issues in queues of Taxi Security team"

    def setup(self):
        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.queues = self._config_getlist("plugins.taxi_security", "queues")
        self.abc_ids = self._config_getlist("plugins.taxi_security", "abc_ids")
        self.slugs = self._config_getlist("plugins.taxi_security", "slugs")

    def main(self):
        self.assignee = self.config.get("plugins.taxi_security", "default_assignee")
        for (abc_id, slug, queue) in zip(self.abc_ids, self.slugs, self.queues):
            shifts = self.abc.get_shifts(abc_id)
            if shifts:
                for shift in shifts:
                    duty_assignee = shift["person"]["login"]
                    shifts_slug = shift["schedule"]["slug"]
                    if shifts_slug == slug:
                        self.assignee = duty_assignee
                        break
                self._walk_queue(queue)

    def _walk_queue(self, queue):
        st_query = (
            "Queue: %s "
            + "Resolution: empty() "
            + "Assignee: empty() "
            + "Tags: !processed "
            + "Created: < now()"
        )
        st_query = st_query % queue
        issues = self.startrek.issues.find(st_query)
        for issue in issues:
            self._process_issue(issue)

    def _process_issue(self, issue):
        if self.assignee:
            if self._check_tracker_summon(issue):
                for linked in issue.links.get_all():
                    if linked.object.key in issue.summary:
                        linked_issue = self.startrek.issues[linked.object.key]
                        linked_issue.followers.append(self.assignee)
                        linked_issue.update(
                            ignore_version_change=True, followers=linked_issue.followers
                        )

            issue.tags.append("processed")
            return issue.update(
                assignee=self.assignee, tags=issue.tags, ignore_version_change=True
            )

    def _check_tracker_summon(self, issue):
        description = issue.description
        if description:
            marker = r"\[Created via e\-mail received from: st\@yandex\-team\.ru\]$"
            matches = re.findall(marker, description)
            links = issue.links.get_all()
            summary_marker = "invites the mailing list to comment"
            verdict = matches and links and (summary_marker in issue.summary)
            return verdict
        return False

    def _close_issue(self, issue, comment):
        tags = list(issue.tags)
        tags.append("auto_closed")
        issue.update(tags=tags, ignore_version_change=True)
        issue = self.startrek.issues[issue.key]
        transition = issue.transitions["close"]
        transition.execute(comment=comment, resolution="won'tFix")
