# -*- coding: utf-8 -*-
import datetime
import logging
import traceback

import startrek_client
from security.yaseclib.abc import Abc
from security.yaseclib.sender import Sender
from security.yaseclib.staff import Staff
from security.yaseclib.wiki import Wiki

from security.c3po.components.core.plugins import BasePlugin


def continue_on_fail(f):
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except:
            logging.exception("[VulnTicketEscalator]\n%s\n" % traceback.format_exc())

    return wrapper


class VulnTicketEscalator(BasePlugin):
    title = u"VulnTicketEscalator"
    desc = (
        u"Escalator of security-related tickets in ST. "
        u"Supposed to be launched once every two weeks."
    )

    def _load_conf(self):
        self.vuln_wiki_cluster = self.config.get(
            "plugins.vuln_ticket_escalator", "vuln_wiki_claster"
        )

        self.soc_wiki_cluster = self.config.get(
            "plugins.vuln_ticket_escalator", "soc_wiki_claster"
        )

        self.vuln_form_id = self.config.get(
            "plugins.vuln_ticket_escalator", "vuln_form_id"
        )

        self.soc_form_id = self.config.get(
            "plugins.vuln_ticket_escalator", "soc_form_id"
        )

        self.soc_wiki_cluster = self.config.get(
            "plugins.vuln_ticket_escalator", "soc_wiki_claster"
        )

        self.nonprocess_tags = self._config_getlist(
            "plugins.vuln_ticket_escalator", "nonprocess_tags"
        )
        self.nonprocess_queues = self._config_getlist(
            "plugins.vuln_ticket_escalator", "nonprocess_queues"
        )

        self.campaign_slug = self.config.get(
            "plugins.vuln_ticket_escalator", "campaign_slug"
        )

        self.notificatees = self._config_getlist(
            "plugins.vuln_ticket_escalator", "notificatee"
        )

        self.notificatees = ["%s@yandex-team.ru" % l for l in self.notificatees]

    def setup(self):
        self._load_conf()

        self.query_vulns = ""
        for queue in self.nonprocess_queues:
            self.query_vulns += "Queue: !%s " % queue

        self.query_vulns += (
            "Security: notEmpty() "
            "Status: !closed "
            "Status: !Production "
            "Tags: !molly "
            '"Voted By": robot-c3po@ '
            '("Security Severity": 4 '
            'OR "Security Severity": 5 '
            'OR "Security Severity": 6) '
            '"Sort By": Priority DESC'
        )

        self.query_soc = "Queue: SECALERTS " "Status: !Closed " "Components: escalate"

        self.wiki = Wiki(
            base_url=self.config.get("wiki", "url"),
            token=self.config.get("wiki", "token"),
        )

        self.startrek = startrek_client.Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.sender = Sender(
            base_url=self.config.get("sender", "url"),
            token=self.config.get("sender", "token"),
            account_slug=self.config.get("sender", "account_slug"),
        )

        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.officers = self._config_getlist(
            "plugins.vuln_ticket_escalator", "officers"
        )
        self.also_directs = self._config_getlist(
            "plugins.vuln_ticket_escalator", "also_directs"
        )

    def main(self):
        # Vuln part
        urls = {}
        report = []
        vuln_tickets = self.walk_st(self.query_vulns)
        for vuln_ticket in vuln_tickets:
            if self._escalate_vuln(vuln_ticket):
                report.append(self._form_report(vuln_ticket))

        if report:
            urls["vuln"] = self.wikify_report(
                report=report,
                wiki_url=self.vuln_wiki_cluster,
                harbinger_id=self.vuln_form_id,
            )
        # SOC part
        soc_tickets = self.walk_st(self.query_soc)
        report = []
        for soc_ticket in soc_tickets:
            report.append(self._form_report(soc_ticket))

        if report:
            urls["soc"] = self.wikify_report(
                report=report,
                wiki_url=self.soc_wiki_cluster,
                harbinger_id=self.soc_form_id,
            )

        if urls:
            self.send_report(urls, self.campaign_slug)

    @continue_on_fail
    def walk_st(self, search_query):
        for_escalation = []
        issues = self.startrek.issues.find(search_query)
        for issue in issues:
            # Filter EX* queues
            if issue.key.startswith("EX"):
                if int(issue.createdAt[:4]) < 2015:
                    continue
            # Begin processing
            if set(issue.tags).intersection(set(self.nonprocess_tags)):
                continue
            for_escalation.append(issue)
        return for_escalation

    def wikify_report(self, report, wiki_url, harbinger_id):
        escalation_grid_structure = [
            {
                "added_column": {
                    "type": "staff",
                    "multiple": False,
                    "title": "Директ",
                }
            },
            {
                "added_column": {
                    "title": "ABC",
                    "type": "string",
                }
            },
            {
                "added_column": {
                    "title": "Тикет",
                    "type": "ticket",
                }
            },
            {
                "added_column": {
                    "title": "Заголовок",
                    "type": "ticket-subject",
                }
            },
            {
                "added_column": {
                    "title": "Исполнитель",
                    "type": "ticket-assignee",
                }
            },
            {
                "added_column": {
                    "title": "Приоритет",
                    "type": "ticket-priority",
                }
            },
            {
                "added_column": {
                    "title": "Статус",
                    "type": "ticket-status",
                }
            },
            {
                "added_column": {
                    "title": "Дата создания",
                    "type": "ticket-createdat",
                }
            },
            {
                "added_column": {
                    "title": "Эскалировать",
                    "markdone": False,
                    "type": "checkbox",
                }
            },
        ]
        date = datetime.date.today()
        report_title = "Форма на эскалацию от %s" % date
        report_url = "%sreport_from%s" % (wiki_url, date)

        grid_title = "Список на эскалацию от %s" % date
        grid_suburl = "grid_of_report_from%s" % date
        grid_url = "%s/%s" % (report_url, grid_suburl)

        text = '{{grid page="/%s" width="100%%" }}\n' "{{forms id=%s}}" % (
            grid_url,
            harbinger_id,
        )
        self.wiki.make_a_grid(
            title=grid_title,
            layout=escalation_grid_structure,
            filling=report,
            url=grid_url,
        )
        self.wiki.write_page(title=report_title, text="", url=report_url)
        self.wiki.write_page(title=report_title, text=text, url=report_url)
        return report_url

    def send_report(self, urls, sender_slug):
        args = {"urls": urls}
        for notificatee in self.notificatees:
            self.sender.send(sender_slug, notificatee, args)

    def _form_report(self, issue):
        lead = None
        summonee = issue.queue.lead.login
        queue_name = issue.queue.key
        abc_service = self.abc.get_service_by_startrek(queue_name)
        if issue.assignee:
            summonee = issue.assignee.login

        if not self._is_yandexoid(summonee):
            if abc_service:
                summonee = self.abc.get_service_responsibles(abc_service["id"])[-1]
                if not self._is_yandexoid(summonee):
                    followers = []
                    if issue.followers:
                        for follower in issue.followers:
                            if self._is_yandexoid(follower.login):
                                if (
                                    self.staff.get_person_department_chief(
                                        follower.login
                                    )
                                    != "sterh"
                                ):
                                    if follower.login != "sterh":
                                        followers.append(follower.login)
                    if followers:
                        summonee = followers[0]
                    else:
                        summonee = "styskin"

        chief_list = self.staff.get_person_chief_list(summonee)
        lead = summonee
        if chief_list[2:]:
            lead = chief_list[2]

        if abc_service:
            abc_service = (
                "https://abc.yandex-team.ru/" "services/%s" % abc_service["slug"]
            )
        else:
            abc_service = ""

        # Temporary fix so tokza can view all these tickets
        tokza = self.startrek.users.get("tokza")
        if tokza not in issue.followers:
            issue.followers.append(tokza)
            issue.followers.append(self.startrek.users.get("sterh"))
            issue.update(followers=issue.followers, ignore_version_change=True)
        return [lead, abc_service, issue.key]

    def _escalate_vuln(self, issue):
        if not self._verify_deadline(issue):
            comments = issue.comments.get_all()
            for comment in comments:
                if comment.createdBy.login in self.officers:
                    summonees = comment.summonees
                    if summonees:
                        assignee_bosses = self.get_assignee_bosses(issue)
                        if set(assignee_bosses) == set(summonees):
                            issue.tags.append("gasecga")
                            issue.update(
                                tags=issue.tags,
                                ignore_version_change=True,
                                params={"notify": "False"},
                            )
                            return True

        return False

    def _verify_deadline(self, issue):
        # Return False if deadline is missed and True if everything's on time
        if issue.deadline:
            today = datetime.datetime.now().date()
            deadline = datetime.datetime.strptime(issue.deadline, "%Y-%m-%d").date()
            if 180 >= (deadline - today).days >= 0:
                return True
        return False

    def _is_yandexoid(self, login):
        return (
            login
            and self.staff.is_yandexoid(login)
            and not self.staff.is_external(login)
        )

    def _get_followers(self, issue):
        result = []
        if issue.followers:
            for follower in issue.followers:
                result.append(follower.login)
        return result

    def get_assignee_bosses(self, issue):
        """Return list of assignee bosses."""
        assignee_bosses = []
        direct = ""
        summonee = []

        comments = issue.comments.get_all()
        if comments:
            comments = comments._data

        assignee = issue.assignee
        author = issue.createdBy
        qlead = issue.queue.lead

        followers_logins = self._get_followers(issue)
        officer_flwr = [
            officer for officer in self.officers if officer in followers_logins
        ]
        assignee_ok = assignee and self._is_yandexoid(assignee.login)
        qlead_ok = qlead and self._is_yandexoid(qlead.login)

        # If there are followers from InfoSec, or maybe the author
        # If there is an assignee and he's still working we summon him
        # Else, if there's a queue lead and he's still working we summon him
        # Else, we summon somebody from the security followers
        # Or the author, duh
        if (officer_flwr or (author.login in self.officers)) and not any(
            u in officer_flwr for u in issue.pendingReplyFrom
        ):
            if assignee_ok:
                summonee = assignee
                assignee_bosses = self.staff.get_person_chief_list(summonee.login)[2:]
            elif qlead_ok:
                summonee = qlead
                assignee_bosses = self.staff.get_person_chief_list(summonee.login)[2:]
            elif author:
                if author.login in self.officers:
                    summonee.append(author)
            else:
                summonee.append(officer_flwr[-1])

            if assignee_bosses:
                direct = assignee_bosses[0]
                assignee_bosses = assignee_bosses[1:]
                also_direct = set(self.also_directs) & set(assignee_bosses)
                if also_direct:
                    direct = also_direct.pop()
                    indent = assignee_bosses.index(direct)
                    assignee_bosses = assignee_bosses[indent:]

        return assignee_bosses
