# -*- coding: utf-8 -*-
import datetime
import logging
import traceback

import startrek_client
from security.yaseclib.sender import Sender
from security.yaseclib.staff import Staff

from security.c3po.components.core.plugins import BasePlugin


def continue_on_fail(f):
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except:
            logging.exception("[VulnTicketReporter]\n%s\n" % traceback.format_exc())

    return wrapper


class VulnTicketReporter(BasePlugin):
    title = u"VulnTicketReporter"
    desc = (
        u"Reporter of security-related tickets in ST. "
        u"Take a pill once every week, get a report "
        u"on tickets with no Security Officers present"
    )

    def setup(self):
        self._load_conf()

        self.strange_tickets = []

        self.startrek = startrek_client.Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.sender = Sender(
            base_url=self.config.get("sender", "url"),
            token=self.config.get("sender", "token"),
            account_slug=self.config.get("sender", "account_slug"),
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

    def main(self):
        self.walk_st()

    def _load_conf(self):
        self.policy = {
            "blocker": int(self.config.get("plugins.vuln_ticket_reporter", "blocker")),
            "critical": int(
                self.config.get("plugins.vuln_ticket_reporter", "critical")
            ),
            "normal": int(self.config.get("plugins.vuln_ticket_reporter", "normal")),
            "minor": int(self.config.get("plugins.vuln_ticket_reporter", "minor")),
            "trivial": int(self.config.get("plugins.vuln_ticket_reporter", "trivial")),
        }

        self.nonprocess_tags = self._config_getlist(
            "plugins.vuln_ticket_reporter", "nonprocess_tags"
        )
        self.nonprocess_queues = self._config_getlist(
            "plugins.vuln_ticket_reporter", "nonprocess_queues"
        )

        self.officers = self._config_getlist("plugins.vuln_ticket_reporter", "officers")

        self.campaign_slug = self.config.get(
            "plugins.vuln_ticket_reporter", "reporter_campaign_slug"
        )

        self.notificatee = "%s@yandex-team.ru" % self.config.get(
            "plugins.vuln_ticket_reporter", "notificatee"
        )

    @continue_on_fail
    def walk_st(self):
        search_query = ""
        for queue in self.nonprocess_queues:
            search_query += "Queue: !%s " % queue

        search_query += (
            "Security: notEmpty() "
            + "Status: !closed "
            + "Status: !Production "
            + "Tags: !molly "
            + "Tags: !crasher "
        )

        issues = self.startrek.issues.find(search_query)
        for issue in issues:
            # Filter EX* queues
            if issue.key.startswith("EX"):
                if int(issue.createdAt[:4]) < 2015:
                    continue

            # Set policy for the issue
            policy = self.policy[issue.priority.key]

            # Begin processing
            if set(issue.tags).intersection(set(self.nonprocess_tags)):
                continue
            elif self._check_date(issue) > policy:
                self._process_issue(issue)

        if self.strange_tickets:
            args = {"tickets": self.strange_tickets}
            self.sender.send(self.campaign_slug, self.notificatee, args)

    @continue_on_fail
    def _process_issue(self, issue):
        """Return True if ticket has to be reported about."""
        followers_logins = self._get_followers(issue)
        author = issue.createdBy.login

        assignee = issue.assignee
        if assignee and self._is_yandexoid(assignee.login):
            assignee = assignee.login
        else:
            assignee = "None"

        officer_flwr = [
            officer for officer in self.officers if officer in followers_logins
        ]

        # If there are no followers or authors from ProdSec,
        # We add a ticket, and some basic info, to the report
        if not (
            officer_flwr or (author in self.officers) or (assignee in self.officers)
        ):
            record = {
                "key": issue.key,
                "summary": issue.summary,
                "assignee": assignee,
                "createdAt": issue.createdAt,
                "priority": issue.priority.key,
            }
            self.strange_tickets.append(record)
            return True

        return False

    def _get_followers(self, issue):
        result = []
        if issue.followers:
            for follower in issue.followers:
                result.append(follower.login)
        return result

    def _check_date(self, issue):
        return (
            datetime.datetime.now().date()
            - datetime.datetime.strptime(issue.updatedAt[:10], "%Y-%m-%d").date()
        ).days

    def _is_yandexoid(self, login):
        return (
            login
            and self.staff.is_yandexoid(login)
            and not self.staff.is_external(login)
        )
