# -*- coding: utf-8 -*-
import datetime
import logging
import traceback
from collections import defaultdict

import requests
from jinja2 import Environment, FileSystemLoader

import startrek_client
from security.c3po.components.core.common import service_mapping
from security.c3po.components.core.plugins import BasePlugin
from security.yaseclib.abc import Abc
from security.yaseclib.gap import Gap
from security.yaseclib.sender import Sender
from security.yaseclib.staff import Staff
from security.yaseclib.statmaster import StatSender
from security.yaseclib.tvm import TVM


def continue_on_fail(f):
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except Exception:
            logging.exception("[VulnTicketReviver]\n%s\n" % traceback.format_exc())

    return wrapper


class VulnTicketReviver(BasePlugin):
    title = u"VulnTicketReviver"
    desc = (
        u"Updater of forgotten security-related tickets in ST. "
        u"Designed to be launched once a day."
    )

    def setup(self):
        self._load_conf()

        self.voter = requests.session()
        self.voter.headers = {
            "Authorization": "OAuth " + self.config.get("st", "token")
        }

        self.for_escalation = []
        self.escalate_mail = {}

        self.summon_count = {}
        self.stat_data = {
            "blocker": 0,
            "critical": 0,
            "normal": 0,
            "minor": 0,
            "trivial": 0,
        }
        self.tvm = TVM(
            client_id=self.config.get("tvm", "client_id"),
            client_secret=self.config.get("tvm", "client_secret"),
            destinations=self._config_getlist("tvm", "destinations"),
        )
        self.calendar_tvm_ticket = self.tvm.get_service_ticket(
            Gap.CALENDAR_YATEAM_TVM_ID
        )

        self.gap = Gap(
            base_url=self.config.get("gap", "url"),
            token=self.config.get("gap", "token"),
            base_calendar_url=self.config.get("calendar", "url"),
            calendar_tvm_service_ticket=self.calendar_tvm_ticket,
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.abc = Abc(
            base_url=self.config.get("abc", "url"),
            token=self.config.get("abc", "token"),
        )

        self.stat_sender = StatSender(
            self.config.get("stat", "robot_login"),
            self.config.get("stat", "robot_password"),
            self.config.get("stat", "url"),
        )

        self.startrek = startrek_client.Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.sender = Sender(
            base_url=self.config.get("sender", "url"),
            token=self.config.get("sender", "token"),
            account_slug=self.config.get("sender", "account_slug"),
        )

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "comment_templates"
            )
        )

        sec_abc_id = self.config.get("plugins.vuln_ticket_reviver", "sec_abc_id")
        shifts = self.abc.get_shifts(sec_abc_id)

        self.coresec_assignee = None
        self.duty_assignees = {}
        for shift in shifts:
            shifts_slug = shift["schedule"]["slug"]
            duty_assignee = shift["person"]["login"]
            if shifts_slug == "coresec_everyday":
                self.core_assignee = duty_assignee
            if duty_assignee in self.duty_assignees:
                self.duty_assignees[duty_assignee].update(
                    service_mapping[shifts_slug]
                )
            else:
                self.duty_assignees[duty_assignee] = service_mapping[shifts_slug]

        # Add Cloud shifts
        shifts_cloud = self.get_shifts_cloud()
        if shifts_cloud:
            cloud_assignee = shifts_cloud[0]
        else:
            cloud_assignee = "axlodin"

        if cloud_assignee in self.duty_assignees:
            self.duty_assignees[cloud_assignee].add("abash")
        else:
            self.duty_assignees[cloud_assignee] = {"abash"}

        # Re-mapping to {boss: assignee}
        self.bosses = defaultdict(set)
        for key, values in self.duty_assignees.items():
            for value in values:
                self.bosses[value] = key

    def main(self):
        if self.gap.is_workday():
            self.walk_st()
        self.tvm.stop()

    def _load_conf(self):
        self.policy = {
            "0": int(self.config.get("plugins.vuln_ticket_reviver", "severity_0")),
            "1": int(self.config.get("plugins.vuln_ticket_reviver", "severity_1")),
            "2": int(self.config.get("plugins.vuln_ticket_reviver", "severity_2")),
            "3": int(self.config.get("plugins.vuln_ticket_reviver", "severity_3")),
            "4": int(self.config.get("plugins.vuln_ticket_reviver", "severity_4")),
            "5": int(self.config.get("plugins.vuln_ticket_reviver", "severity_5")),
            "6": int(self.config.get("plugins.vuln_ticket_reviver", "severity_6")),
        }

        self.no_remind_tag = self.config.get("plugins.vuln_ticket_reviver", "no_remind")

        self.open_tickets_link = self.config.get(
            "plugins.vuln_ticket_reviver", "open_vuln_tickets"
        )

        self.escalwarn_campaign_slug = self.config.get(
            "plugins.vuln_ticket_reviver", "escalwarn_campaign_slug"
        )

        self.divinities = self._config_getlist(
            "plugins.vuln_ticket_reviver", "divinities"
        )
        self.also_directs = self._config_getlist(
            "plugins.vuln_ticket_reviver", "also_directs"
        )
        self.officers = self._config_getlist("plugins.vuln_ticket_reviver", "officers")
        self.nonprocess_tags = self._config_getlist(
            "plugins.vuln_ticket_reviver", "nonprocess_tags"
        )
        self.nonprocess_queues = self._config_getlist(
            "plugins.vuln_ticket_reviver", "nonprocess_queues"
        )

    def walk_st(self):
        search_query = ""
        for queue in self.nonprocess_queues:
            search_query += "Queue: !%s " % queue

        search_query += (
            "Security: notEmpty() "
            "Status: !closed "
            "Status: !resolved "
            "Status: !Production "
            "((Queue:!SECALERTS) OR "
            "((Type: Task OR Type: Bug) Queue: SECALERTS)) "
        )
        # search_query = "Queue: SECROBOTS Security: notEmpty() " + \
        #                "Status: open, in_progress, need_info"
        issues = self.startrek.issues.find(search_query)
        self.for_escalation = []
        today = datetime.datetime.now().date()
        for issue in issues:

            # Form report data
            self.stat_data[issue.priority.key] += 1

            # Begin processing
            if set(issue.tags).intersection(set(self.nonprocess_tags)):
                continue

            elif self.no_remind_tag in issue.tags:
                self._no_remind(issue)

            elif issue.deadline:
                deadline = datetime.datetime.strptime(issue.deadline, "%Y-%m-%d").date()
                if deadline < today:
                    margin = (self.policy[issue.securitySeverity] - 1) // 3
                    updated = datetime.datetime.strptime(
                        issue.updatedAt[:10], "%Y-%m-%d"
                    ).date()
                    if margin:
                        if (today - updated).days % self.policy[
                            issue.securitySeverity
                        ] > margin:
                            self._new_process_issue(issue)

                elif issue.securitySeverity > "3":
                    if (deadline - today).days == 2:
                        self.escalation_warning(issue)
                    elif deadline == today:
                        voters = {a.login for a in issue.votedBy}
                        if self.startrek.myself.login not in voters:
                            self.vote_for_issue(issue)
                        assignee_bosses = self.get_assignee_bosses(issue)
                        template = "duty_informer.tpl"
                        template_args = {"assignee_bosses": assignee_bosses}
                        t = self.env.get_template(template)
                        text = t.render(**template_args)

                        boss_list = self.staff.get_person_chief_list(
                            issue.queue.lead.login
                        )
                        bosses_candidates = [
                            boss for boss in boss_list if boss in self.bosses
                        ]
                        boss = bosses_candidates.pop()
                        officer = self.bosses[boss]
                        security_summonee = None
                        if self.gap.get_working_today(officer):
                            security_summonee = officer

                        issue.comments.create(summonees=security_summonee, text=text)

        self.stat_sender.post_to_stat(name=self.open_tickets_link, data=self.stat_data)

        if self.escalate_mail:
            for officer, tickets in self.escalate_mail.items():
                args = {"tickets": tickets}
                self.sender.send(
                    self.escalwarn_campaign_slug, "%s@yandex-team.ru" % officer, args
                )

    def escalation_warning(self, issue):
        # Warn Security Team before summoning a Direct
        author = issue.createdBy

        assignee = issue.assignee
        assignee_ok = assignee and self._is_yandexoid(assignee.login)

        followers_logins = self._get_followers(issue)
        officer_flwr = [
            officer for officer in self.officers if officer in followers_logins
        ]
        mail_to = list({author.login} & set(self.officers) | set(officer_flwr))
        for officer in mail_to:
            if officer not in self.escalate_mail:
                self.escalate_mail[officer] = []

            record = {
                "key": issue.key,
                "summary": issue.summary,
                "assignee": assignee.login if assignee_ok else None,
                "createdAt": issue.createdAt,
                "priority": issue.priority.key,
            }
            self.escalate_mail[officer].append(record)

    def get_assignee_bosses(self, issue):
        """Return list of assignee bosses."""
        assignee_bosses = []
        direct = ""
        summonee = []

        comments = issue.comments.get_all()
        if hasattr(comments, 'data'):
            comments = comments.data

        assignee = issue.assignee
        author = issue.createdBy
        qlead = issue.queue.lead

        followers_logins = self._get_followers(issue)
        officer_flwr = [
            officer for officer in self.officers if officer in followers_logins
        ]
        assignee_ok = assignee and self._is_yandexoid(assignee.login)
        qlead_ok = qlead and self._is_yandexoid(qlead.login)

        # If there are followers from InfoSec, or maybe the author
        # If there is an assignee and he's still working we summon him
        # Else, if there's a queue lead and he's still working we summon him
        # Else, we summon somebody from the security followers
        # Or the author, duh
        if (officer_flwr or (author.login in self.officers)) and not any(
            u in officer_flwr for u in issue.pendingReplyFrom
        ):
            if assignee_ok:
                summonee = assignee
                assignee_bosses = self.staff.get_person_chief_list(summonee.login)[2:]
            elif qlead_ok:
                summonee = qlead
                assignee_bosses = self.staff.get_person_chief_list(summonee.login)[2:]
            elif author:
                if author.login in self.officers:
                    summonee.append(author)
            else:
                summonee.append(officer_flwr[-1])

            if assignee_bosses:
                direct = assignee_bosses[0]
                assignee_bosses = assignee_bosses[1:]
                also_direct = set(self.also_directs) & set(assignee_bosses)
                if also_direct:
                    direct = also_direct.pop()
                    indent = assignee_bosses.index(direct)
                    assignee_bosses = assignee_bosses[indent:]

        return assignee_bosses

    def _new_process_issue(self, issue):
        """Return True if ticket has to be escalated."""
        assignee_bosses = []
        direct = ""
        escalated = False
        summonee = []

        template = "summon_vuln_fix.tpl"

        comments = list(issue.comments.get_all())

        assignee = issue.assignee
        author = issue.createdBy
        qlead = issue.queue.lead

        followers_logins = self._get_followers(issue)
        officer_flwr = [
            officer for officer in self.officers if officer in followers_logins
        ]
        assignee_ok = assignee and self._is_yandexoid(assignee.login)
        qlead_ok = qlead and self._is_yandexoid(qlead.login)

        # If there are followers from InfoSec, or maybe the author
        # If there is an assignee and he's still working we summon him
        # Else, if there's a queue lead and he's still working we summon him
        # Else, we summon somebody from the security followers
        # Or the author, duh
        if (officer_flwr or (author.login in self.officers)) and not any(
            u in officer_flwr for u in issue.pendingReplyFrom
        ):
            if assignee_ok:
                summonee = assignee
                assignee_bosses = self.staff.get_person_chief_list(summonee.login)[2:]
            elif qlead_ok:
                summonee = qlead
                assignee_bosses = self.staff.get_person_chief_list(summonee.login)[2:]
            elif author:
                if author.login in self.officers:
                    summonee.append(author)
            else:
                summonee.append(officer_flwr[-1])

            if assignee_bosses:
                direct = assignee_bosses[0]
                assignee_bosses = assignee_bosses[1:]
                also_direct = set(self.also_directs) & set(assignee_bosses)
                if also_direct:
                    direct = also_direct.pop()
                    indent = assignee_bosses.index(direct)
                    assignee_bosses = assignee_bosses[indent:]

            if len(comments) > 0:
                # But, if there are some comments
                # We check if author of last comment is a security officer
                # If an officer had summoned somebody, we summon them
                last_comment = comments[-1]
                if last_comment.createdBy.login in self.officers:
                    if hasattr(last_comment, "summonees"):
                        new_summonees = []
                        for candidate in last_comment.summonees:
                            if self._is_yandexoid(candidate.login):
                                new_summonees.append(candidate)
                        if new_summonees:
                            summonee = last_comment.summonees

                comments_count = self._count_my_comments(issue)

                # Clean up the ticket, if there's too much summons
                if comments_count > 4:
                    for comment in reversed(comments):
                        if comment.createdBy.login == self.startrek.myself.login:
                            comment.delete()
                            break

                # If the escalation condition passes
                if summonee == assignee or summonee == qlead:
                    # Summon bosses only for critical issues
                    if issue.securitySeverity > "3":
                        summonee = [summonee]
                        summonee.extend(assignee_bosses[-2 * comments_count:])

            self._summon(issue, summonee, template, {"summon_security": False})

        return escalated

    def _summon(self, issue, users, template, template_args=None):
        to_summon = []
        if not isinstance(users, list):
            users = [users]
        for user in users:
            if isinstance(user, str) or isinstance(user, bytes):
                user = self.startrek.users.get(user)

            # Only humans are allowed
            if user.login.startswith(("robot-", "zomb-")):
                continue

            # Some people just don't like to be bothered
            if user.login in self.divinities:
                user = self.startrek.users.get("sterh")

            # Load balancing
            if user.login in self.summon_count:
                self.summon_count[user.login] += 1
            else:
                self.summon_count[user.login] = 1

            if self.summon_count[user.login] < 3 and self.gap.get_working_today(
                user.login
            ):
                to_summon.append(user)

        if to_summon:
            to_summon = list(set(to_summon))
            t = self.env.get_template(template)
            text = t.render(**template_args)
            issue.comments.create(text=text, summonees=to_summon)
            return True

    def _count_my_comments(self, issue):
        comments = issue.comments.get_all()
        summons = 0
        for comment in comments:
            if comment.createdBy.login == self.startrek.myself.login:
                summons += 1

        return summons

    def _no_remind(self, issue):
        if self._check_update_date(issue) > 45:
            new_tags = list(issue.tags)
            new_tags.remove(self.no_remind_tag)
            issue.update(tags=new_tags)

    def _get_followers(self, issue):
        result = []
        if issue.followers:
            for follower in issue.followers:
                result.append(follower.login)
        return result

    def _check_update_date(self, issue):
        today = datetime.datetime.now().date()
        update_date = datetime.datetime.strptime(
            issue.updatedAt[:10], "%Y-%m-%d"
        ).date()
        return (today - update_date).days

    def _verify_deadline(self, issue):
        # Return False if deadline is missed and True if everything's on time
        if issue.deadline:
            today = datetime.datetime.now().date()
            deadline = datetime.datetime.strptime(issue.deadline, "%Y-%m-%d").date()
            if (deadline - today).days >= 0:
                return True
        return False

    def get_for_escalation(self):
        return self.for_escalation

    def _is_yandexoid(self, login):
        return (
            login
            and self.staff.is_yandexoid(login)
            and not self.staff.is_external(login)
        )

    def vote_for_issue(self, issue):
        self.voter.post("https://st-api.yandex-team.ru/v2/issues/%s/_vote" % issue.key)

    def get_shifts_cloud(self):
        query = (
            "Queue: CLOUDDUTY "
            "Components: 47232 "
            "Status:!closed "
            '"Sort By": Created ASC'
        )
        issues = self.startrek.issues.find(query)
        result = []

        for issue in issues:
            if issue.assignee:
                result.append(issue.assignee.login)

        return result
