from datetime import date, timedelta

from jinja2 import Environment, FileSystemLoader
from startrek_client import Startrek
from security.yaseclib.sender import Sender
from security.yaseclib.staff import Staff

from security.c3po.components.core.plugins import BasePlugin


class WelcomeRecruits(BasePlugin):
    title = "Welcome letter to all recruits (except external)"
    desc = "Создание тикетов для прохождения обучения по ИБ новыми сотрудниками"

    def setup(self):
        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.sender = Sender(
            base_url=self.config.get("sender", "url"),
            token=self.config.get("sender", "token"),
            account_slug=self.config.get("sender", "account_slug"),
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "templates"
            )
        )

        self.all_campaign_slug = self.config.get(
            "plugins.welcome_recruits", "sender_campaign_for_all"
        )

    def main(self):
        yesterday_iso = (date.today() - timedelta(1)).isoformat()
        t = self.env.get_template("ticket_for_all.tpl")

        today = date.today()
        deadline = today + timedelta(days=90)

        all_recruits = self.staff.get_persons_by_depart_url(
            "yandex", yesterday_iso, limit=1000
        )

        for user in all_recruits:
            body = t.render(
                name_ru=user["name"]["first"]["ru"],
                date=deadline.strftime("%d.%m.%Y"),
                name_en=user["name"]["first"]["en"],
            )
            if (user["official"]["affiliation"]) == "yandex":
                boss_list = self.staff.get_person_chief_list(user["login"])
                created_issue = self.startrek.issues.create(
                    queue="AWARENESS",
                    summary="Обязательные курсы по информационной безопасности",
                    assignee=user["login"],
                    description=body,
                    components="Recruits",
                    followers=boss_list[-1],
                    deadline=deadline.isoformat(),
                )

                to_email = "{}@yandex-team.ru".format(user["login"])
                args = {
                    "name_ru": user["name"]["first"]["ru"],
                    "ticket": created_issue.key,
                    "name_en": user["name"]["first"]["en"],
                }
                self.sender.send(self.all_campaign_slug, to_email, args)
