import logging
import traceback
from datetime import date, timedelta, datetime

from jinja2 import Environment, FileSystemLoader
from startrek_client import Startrek
from security.yaseclib.moe import Moe
from security.yaseclib.staff import Staff
from security.yaseclib.tvm import TVM

from security.c3po.components.core.plugins import BasePlugin


def continue_on_fail(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except:
            logging.exception(
                "[WelcomeRecruitsReminder]\n%s\n" % traceback.format_exc()
            )

    return wrapper


class WelcomeRecruitsReminder(BasePlugin):
    title = "Welcome letter to all recruits (except external)"
    desc = "Напоминание новым сотрудникам о необходимости пройти обучение из тикетов welcome_recruits"

    def setup(self):
        self.tvm = TVM(
            client_id=self.config.get("tvm_c3po", "client_id"),
            client_secret=self.config.get("tvm_c3po", "client_secret"),
            destinations=self._config_getlist("tvm_c3po", "destinations"),
        )

        self.startrek = Startrek(
            useragent=self.config.get("st", "ua"),
            base_url=self.config.get("st", "url"),
            token=self.config.get("st", "token"),
        )

        self.env = Environment(
            loader=FileSystemLoader(
                self.resource_path() / "templates"
            )
        )

        self.staff = Staff(
            base_url=self.config.get("staff", "url"),
            token=self.config.get("staff", "token"),
        )

        self.moe_tvm_ticket = self.tvm.get_service_ticket(Moe.DEFAULT_MOE_TVM_ID)

        self.moe = Moe(tvm_ticket=self.moe_tvm_ticket)

        self.all_default_courses = self._config_getlist(
            "plugins.welcome_recruits_reminder", "all_default_courses"
        )

        self.dev_course = self._config_getlist(
            "plugins.welcome_recruits_reminder", "dev_course"
        )

    @continue_on_fail
    def _pinger(self, template_ping, template_close, template_reopen):

        st_query = (
            "Queue: AWARENESS "
            "Status: !closed "
            "Assignee: notEmpty() "
            "Tags: !english "
            "Tags: !no_remind "
            '"Sort By": Key ASC'
        )

        issues = self.startrek.issues.find(st_query)

        for issue in issues:
            if (
                issue.status.key in ["requestApprove"]
                or self._check_update_date(issue) > 7
            ):
                deadline = None
                if issue.deadline:
                    deadline = datetime.strptime(issue.deadline, "%Y-%m-%d")
                name = issue.assignee.firstName
                components = issue.components
                login = issue.assignee.login
                if self.staff.get_person_info(login, field="official.is_dismissed"):
                    for status in issue.transitions.get_all():
                        if status.id == "closed":
                            transition = issue.transitions["closed"]
                            transition.execute(
                                resolution="fixed", tags=issue.tags + ["dismissed"]
                            )
                            break
                        elif status.id == "close":
                            transition = issue.transitions["close"]
                            transition.execute(
                                resolution="fixed", tags=issue.tags + ["dismissed"]
                            )
                            break
                else:
                    comp_names = [c.name for c in components]
                    course_list = list()

                    if "Recruits" in comp_names:
                        course_list.extend(self.all_default_courses)
                    if "DevCourse" in comp_names:
                        course_list.extend(self.dev_course)

                    result = dict()
                    for course in course_list:
                        if course == self.dev_course[0]:
                            course_result = self.moe.get_user_result_by_course(
                                course, login, 70.0
                            )
                        else:
                            course_result = self.moe.get_user_result_by_course(
                                course, login, 80.0
                            )
                        if not course_result:
                            result.update({course: False})
                        else:
                            result.update({course: course_result[0][1]})

                    if not deadline:
                        today = date.today()
                        deadline = today + timedelta(days=90)
                        issue.update(deadline=deadline.isoformat())

                    if all(x for x in result.values()):
                        c = self.env.get_template(template_close)
                        comment = c.render(name=name)

                        for status in issue.transitions.get_all():
                            if status.id == "closed":
                                transition = issue.transitions["closed"]
                                transition.execute(comment=comment, resolution="fixed")
                                break
                            elif status.id == "close":
                                transition = issue.transitions["close"]
                                transition.execute(comment=comment, resolution="fixed")
                                break
                    elif issue.status.key in ["requestApprove"]:
                        c = self.env.get_template(template_reopen)
                        comment = c.render(name=name, result=result)
                        transition = issue.transitions["open"]
                        transition.execute()
                        issue.comments.create(text=comment, summonees=[issue.assignee])
                    else:
                        c = self.env.get_template(template_ping)
                        comment = c.render(
                            name=name, date=deadline.strftime("%d.%m.%Y"), result=result
                        )
                        issue.comments.create(text=comment, summonees=[issue.assignee])

    @staticmethod
    def _check_update_date(issue):
        today = date.today()
        update_date = datetime.strptime(issue.updatedAt[:10], "%Y-%m-%d").date()
        return (today - update_date).days

    def main(self):
        template_ping = "ping_text.tpl"
        template_close = "close_text.tpl"
        template_reopen = "reopen_text.tpl"
        self._pinger(template_ping, template_close, template_reopen)
        self.tvm.stop()
