package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/csp-report/internal/app"
	"a.yandex-team.ru/security/csp-report/internal/config"
	"a.yandex-team.ru/security/csp-report/internal/pprof"
)

var (
	configPath string
)

func newLogger(logLvl string) (log.Structured, error) {
	lvl, err := log.ParseLevel(logLvl)
	if err != nil {
		return nil, fmt.Errorf("failed to parse log level %q: %w", logLvl, err)
	}

	l, err := zap.NewDeployLogger(lvl)
	if err != nil {
		return nil, fmt.Errorf("failed to create logger: %w", err)
	}

	l.Info("logger initialized", log.String("level", lvl.String()))
	return l, nil
}

func main() {
	maxprocs.AdjustYP()
	flag.StringVar(&configPath, "config", "", "config path (e.g. /csp-report.toml)")
	flag.Parse()

	cfg, err := config.NewConfig(configPath)
	if err != nil {
		_, _ = fmt.Fprintln(os.Stderr, err.Error())
		os.Exit(1)
	}

	logger, err := newLogger(cfg.LogLevel)
	if err != nil {
		_, _ = fmt.Fprintln(os.Stderr, err.Error())
		os.Exit(1)
	}

	if buildinfo.Info.ProgramVersion != "" {
		logger.Info("build info",
			log.String("revision", buildinfo.Info.ArcadiaSourceRevision),
			log.String("date", buildinfo.Info.Date),
		)
	}

	instance, err := app.New(cfg, app.WithLogger(logger), app.WithTvm())
	if err != nil {
		logger.Fatal("failed to create app instance", log.Error(err))
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGUSR1)

	errChan := make(chan error, 1)
	go func() {
		logger.Info("starting HTTP application")
		if err := instance.Start(); err != nil {
			errChan <- err
		}
	}()

	pprofSrv := pprof.NewServer(9000)
	go func() {
		_ = pprofSrv.ListenAndServe()
	}()
	defer func() {
		_ = pprofSrv.Shutdown(context.TODO())
	}()

	select {
	case <-stopChan:
		logger.Info("shutting down")
		if err := instance.Shutdown(context.TODO()); err != nil {
			logger.Fatal("failed to shutdown app instance", log.Error(err))
		}
	case err := <-errChan:
		logger.Fatal("shit happens", log.Error(err))
	}
}
