package main

import (
	"bytes"
	"context"
	"encoding/json"
	"flag"
	"fmt"
	"os"
	"time"

	"github.com/klauspost/compress/zstd"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func fatalf(format string, args ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, format+"\n", args...)
	os.Exit(1)
}

func main() {
	var (
		endpoint string
		port     int
		consumer string
		topic    string
		maxMsgs  int
		timeout  time.Duration
	)
	flag.StringVar(&endpoint, "endpoint", "", "LB endpoint")
	flag.IntVar(&port, "port", 0, "LB port")
	flag.StringVar(&consumer, "consumer", "", "LB consumer")
	flag.StringVar(&topic, "topic", "", "LB topic")
	flag.IntVar(&maxMsgs, "max-msgs", 0, "maximum msgs count")
	flag.DurationVar(&timeout, "timeout", 30*time.Second, "whole timeout")
	flag.Parse()

	zapCFG := zap.ConsoleConfig(log.DebugLevel)
	zapCFG.OutputPaths = []string{"stderr"}
	l, err := zap.New(zapCFG)
	if err != nil {
		fatalf("can't create logger: %v", err)
	}

	c := persqueue.NewReader(persqueue.ReaderOptions{
		Endpoint:             endpoint,
		Port:                 port,
		Consumer:             consumer,
		Logger:               corelogadapter.New(l),
		Topics:               []persqueue.TopicInfo{{Topic: topic}},
		MaxReadSize:          8 * 1024, // 100 kb
		MaxReadMessagesCount: 2048,
		// zstd was broken in lb
		DecompressionDisabled: true,
		CommitsDisabled:       true,
		RetryOnFailure:        true,
	})

	// TODO(buglloc): ugly
	ctx, cancel := context.WithTimeout(context.Background(), timeout)
	defer cancel()

	_, err = c.Start(ctx)
	if err != nil {
		fatalf("can't start lb reader: %v", err)
	}

	reader, err := zstd.NewReader(nil, zstd.WithDecoderConcurrency(1))
	if err != nil {
		fatalf("can't start zstd reader: %v", err)
	}

	var messages [][]byte
loop:
	for msg := range c.C() {
		l.Info("new msg", log.Any("msg", msg))
		switch v := msg.(type) {
		case *persqueue.Data:
			for _, b := range v.Batches() {
				for _, m := range b.Messages {
					data, err := reader.DecodeAll(m.Data, nil)
					if err != nil {
						fatalf("can't decode lb data: %v", err)
					}

					l.Info("got report", log.Binary("report", data))
					messages = append(messages, bytes.Split(data, []byte{'\n'})...)
					if len(messages) >= maxMsgs {
						break loop
					}
				}
			}

		default:
		}
	}

	c.Shutdown()
	<-c.Closed()
	if err := ctx.Err(); err != nil {
		fatalf("shit happens: %v", err)
	}

	err = json.NewEncoder(os.Stdout).Encode(messages)
	if err != nil {
		fatalf("can't encode messages: %v", err)
	}
}
