package app

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"os"
	"time"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/httputil/middleware/recovery"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/csp-report/internal/config"
	"a.yandex-team.ru/security/csp-report/internal/logbroker"
)

var ErrShutdownTimeout = errors.New("shutdown timed out")

type App struct {
	ctx             context.Context
	ctxShutdown     context.CancelFunc
	httpShutdownTTL time.Duration
	lbShutdownTTL   time.Duration
	reportWriteTTL  time.Duration
	maxBodySize     int64
	logger          log.Structured
	tvm             tvm.Client
	lb              *logbroker.LogBroker
	shutdownChan    chan struct{}
	https           http.Server
}

func New(cfg *config.Config, opts ...Option) (*App, error) {
	ctx, cancel := context.WithCancel(context.Background())

	app := &App{
		ctx:             ctx,
		ctxShutdown:     cancel,
		httpShutdownTTL: cfg.App.ShutdownTimeout,
		lbShutdownTTL:   cfg.Logbroker.ShutdownTimeout,
		reportWriteTTL:  cfg.App.ReportWriteTimeout,
		maxBodySize:     cfg.App.MaxBodySize,
		logger:          &nop.Logger{},
		shutdownChan:    make(chan struct{}),
	}
	app.https = http.Server{
		Addr:    cfg.App.Addr,
		Handler: app.router(),
		//ReadHeaderTimeout: cfg.App.ReadTimeout,
		//WriteTimeout:      cfg.App.WriteTimeout,
		//IdleTimeout:       cfg.App.IdleTimeout,
	}

	for _, opt := range opts {
		if err := opt(app); err != nil {
			cancel()
			return nil, err
		}
	}

	var err error
	app.lb, err = logbroker.New(ctx, logbroker.Config{
		LogbrokerConfig: cfg.Logbroker,
		TvmClient:       app.tvm,
		Logger:          app.logger,
		SourceEpoch:     os.Getenv("SOURCE_EPOCH"),
	})

	if err != nil {
		cancel()
		return nil, fmt.Errorf("failed to create logbroker client: %w", err)
	}

	if err := app.lb.Start(); err != nil {
		cancel()
		return nil, fmt.Errorf("failed to start logbroker client: %w", err)
	}

	return app, nil
}

func (a *App) Start() error {
	defer close(a.shutdownChan)
	defer a.lb.Shutdown()

	return a.https.ListenAndServe()
}

func (a *App) Shutdown(ctx context.Context) error {
	defer a.ctxShutdown()

	// grateful shutdown server
	httpCtx, cancel := context.WithTimeout(ctx, a.httpShutdownTTL)
	defer cancel()
	if err := a.https.Shutdown(httpCtx); err != nil {
		select {
		case <-httpCtx.Done():
			// timed out
			return ErrShutdownTimeout
		default:
		}
		return err
	}

	// wait logbroker writer
	lbCtx, cancel := context.WithTimeout(ctx, a.lbShutdownTTL)
	defer cancel()
	select {
	case <-a.shutdownChan:
		// completed normally
		return nil
	case <-lbCtx.Done():
		// timed out
		return ErrShutdownTimeout
	}
}

func (a *App) router() http.Handler {
	r := chi.NewRouter()
	r.Use(recovery.New(recovery.WithLogger(a.logger)))

	r.Post("/csp", a.cspHandler)

	r.Get("/unistat", func(rw http.ResponseWriter, req *http.Request) {
		stat := a.lb.FlushStat()
		err := json.NewEncoder(rw).Encode([][]interface{}{
			{"memq_summ", stat.MemQSize},
			{"mem_inflight_summ", stat.MemStat.Inflight},
			{"mem_errors_summ", stat.MemStat.Errors},
			{"mem_written_summ", stat.MemStat.Written},

			{"diskq_summ", stat.DiskQSize},
			{"disk_inflight_summ", stat.DiskStat.Inflight},
			{"disk_errors_summ", stat.DiskStat.Errors},
			{"disk_written_summ", stat.DiskStat.Written},
		})

		if err != nil {
			a.logger.Error("failed to encode unistat data", log.Error(err))
		}
	})

	r.Get("/ping", func(rw http.ResponseWriter, req *http.Request) {
		_, _ = rw.Write([]byte("pong"))
	})

	r.Get("/", func(rw http.ResponseWriter, req *http.Request) {
		_, _ = rw.Write([]byte("secret area"))
	})

	return r
}
