package app

import (
	"context"
	"io/ioutil"
	"net/http"

	"github.com/valyala/fastjson"

	"a.yandex-team.ru/library/go/core/log"
)

const (
	jsonCtStr       = "application/json"
	cspCtStr        = "application/csp-report"
	ctHeaderStr     = "Content-Type"
	userIPHeaderStr = "X-Forwarded-For-Y"
	uaHeaderStr     = "User-Agent"
	originStr       = "Origin"
	corsHeaderStr   = "Access-Control-Allow-Origin"
	corsValueStr    = "*"
)

func (a *App) cspHandler(rw http.ResponseWriter, req *http.Request) {
	defer func() {
		rw.Header().Set(corsHeaderStr, corsValueStr)
		rw.WriteHeader(http.StatusOK)
	}()

	body, err := ioutil.ReadAll(http.MaxBytesReader(rw, req.Body, a.maxBodySize))
	if err != nil {
		a.logger.Warn("failed to read request body", log.String("uri", req.RequestURI), log.Error(err))
		return
	}

	if len(body) == 0 {
		// Just skip it
		return
	}

	contentType := req.Header.Get(ctHeaderStr)
	if contentType == jsonCtStr || contentType == cspCtStr {
		// ugly hack
		if err := fastjson.ValidateBytes(body); err != nil {
			a.logger.Warn("broken json", log.String("uri", req.RequestURI), log.Error(err))
			return
		}
	}

	ctx, cancel := context.WithTimeout(req.Context(), a.reportWriteTTL)
	defer cancel()

	err = a.lb.WriteReport(
		ctx,
		req.Header.Get(userIPHeaderStr),
		req.RequestURI,
		req.Header.Get(uaHeaderStr),
		req.Header.Get(originStr),
		body,
	)
	if err != nil && ctx.Err() == nil {
		a.logger.Error("failed to write report", log.Error(err))
	}
}
