package logbroker

import (
	"context"
	"fmt"
	"os"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/csp-report/internal/config"
	"a.yandex-team.ru/security/csp-report/internal/logbroker/diskqueue"
	"a.yandex-team.ru/security/csp-report/internal/logbroker/message"
)

type (
	Config struct {
		config.LogbrokerConfig
		TvmClient   tvm.Client
		Logger      log.Structured
		SourceEpoch string
		SourceID    string
	}

	LogBroker struct {
		logger  log.Structured
		memQ    chan *message.Message
		diskQ   *diskqueue.DiskQueue
		writers []*lightWriter
	}
)

func New(ctx context.Context, cfg Config) (*LogBroker, error) {
	if cfg.SourceID == "" {
		var err error
		cfg.SourceID, err = os.Hostname()
		if err != nil || cfg.SourceID == "" {
			cfg.SourceID = uuid.Must(uuid.NewV4()).String()
		}

		cfg.SourceID = fmt.Sprintf("%s:%s", cfg.SourceEpoch, cfg.SourceID)
	}

	lb := &LogBroker{
		logger: cfg.Logger,
		memQ:   make(chan *message.Message, cfg.QueueSize),
		diskQ: diskqueue.MustNew(
			cfg.DBPath,
			diskqueue.WithLogger(cfg.Logger),
			diskqueue.WithMaxBytesPerFile(100<<(10*2)),
			diskqueue.WithMaxMsgSize(64<<10),
		),
	}

	for i := 0; i < cfg.Writers; i++ {
		memWriter, err := newLightWriter(ctx, fmt.Sprintf("memory_%d", i), cfg, lb.memQ)
		if err != nil {
			lb.Shutdown()
			return nil, fmt.Errorf("failed to create memory writer: %w", err)
		}

		lb.writers = append(lb.writers, memWriter)
	}

	diskWriter, err := newLightWriter(ctx, "disk", cfg, lb.diskQ.ReadChan())
	if err != nil {
		lb.Shutdown()
		return nil, fmt.Errorf("failed to create memory writer: %w", err)
	}

	// TODO(buglloc): you could better
	lb.diskQ.SetSeqNo(diskWriter.initialSeqNo)
	lb.writers = append(lb.writers, diskWriter)
	return lb, nil
}

func (l *LogBroker) WriteReport(
	ctx context.Context,
	addr,
	requestURI,
	userAgent,
	origin string,
	content []byte,
) (err error) {

	msg := message.AcquireMsg()
	msg.Addr = addr
	msg.RequestURI = requestURI
	msg.UserAgent = userAgent
	msg.Origin = origin
	msg.Content = content
	msg.TS = time.Now()
	msg.SeqNo = 0

	select {
	case l.memQ <- msg:
		// success!
	case <-ctx.Done():
		// that's fine
	default:
		err = l.diskQ.Put(ctx, msg)
	}
	return
}

func (l *LogBroker) FlushStat() LbStat {
	out := LbStat{
		MemQSize:  len(l.memQ),
		DiskQSize: int(l.diskQ.Size()),
		// disk writer are always the last
		DiskStat: l.writers[len(l.writers)-1].FlushStat(),
	}

	for i := 0; i < len(l.writers)-1; i++ {
		out.MemStat.Merge(l.writers[i].FlushStat())
	}
	return out
}

func (l *LogBroker) Start() error {
	for _, w := range l.writers {
		w.Start()
	}

	return nil
}

func (l *LogBroker) Shutdown() {
	// first of all - close all of our channels
	if l.diskQ != nil {
		err := l.diskQ.Close()
		if err != nil {
			l.logger.Error("failed to close diskQ", log.Error(err))
		}
	}

	if l.memQ != nil {
		close(l.memQ)
	}

	// now wait writers
	for _, w := range l.writers {
		if err := w.Shutdown(context.Background()); err != nil {
			l.logger.Error("failed to close writer", log.String("writer", w.source), log.Error(err))
		}
	}
}
