package message

import (
	"bytes"
	"encoding/gob"
	"io"
	"time"
)

const (
	timeFormat = "2006-01-02 15:04:05 -0700"
)

var (
	prefixStr     = []byte("tskv\ttskv_format=csp-log\t")
	addrKeyStr    = []byte("addr=")
	pathKeyStr    = []byte("path=")
	uaKeyStr      = []byte("user-agent=")
	originKeyStr  = []byte("origin=")
	contentKeyStr = []byte("content=")
	tsKeyStr      = []byte("timestamp=")
	tzKeyStr      = []byte("timezone=")
)

type Message struct {
	Addr       string
	RequestURI string
	UserAgent  string
	Origin     string
	Content    []byte
	TS         time.Time
	SeqNo      uint64
}

func (m *Message) WithSeqNo(seqNo uint64) *Message {
	m.SeqNo = seqNo
	return m
}

func (m *Message) WriteTo(buf *bytes.Buffer) {
	// Prefix
	buf.Write(prefixStr)

	// Fields
	buf.Write(addrKeyStr)
	_ = tskvWriteEscaped(buf, m.Addr)
	buf.WriteRune('\t')

	buf.Write(pathKeyStr)
	_ = tskvWriteEscaped(buf, m.RequestURI)
	buf.WriteRune('\t')

	buf.Write(uaKeyStr)
	_ = tskvWriteEscaped(buf, m.UserAgent)
	buf.WriteRune('\t')

	buf.Write(originKeyStr)
	_ = tskvWriteEscaped(buf, m.Origin)
	buf.WriteRune('\t')

	buf.Write(contentKeyStr)
	tskvWriteEscapedBytes(buf, m.Content)
	buf.WriteRune('\t')

	timeStr := m.TS.Format(timeFormat)
	buf.Write(tsKeyStr)
	_ = tskvWriteEscaped(buf, timeStr[:19])
	buf.WriteRune('\t')

	buf.Write(tzKeyStr)
	_ = tskvWriteEscaped(buf, timeStr[20:])
}

func Marshal(m *Message, data io.Writer) error {
	// TODO(buglloc): lz4/zstd
	enc := gob.NewEncoder(data)
	return enc.Encode(m)
}

func Unmarshal(data io.Reader) (*Message, error) {
	// TODO(buglloc): lz4/zstd
	m := new(Message)
	err := gob.NewDecoder(data).Decode(m)
	return m, err
}
