from __future__ import unicode_literals, absolute_import, print_function

import os
import uuid
import psutil
import json
import subprocess
import datetime

from app import db
from app import utils
from app.settings import STATE_PENDING, STATE_IN_PROGRESS
from app.settings import STATE_CANCELED, STATE_FINISHED

class BaseEngine(object):
    def __init__(self):
        self.engine = None # must be set on child classes
        self.process_results_on_cancel = False

    def create_task(self, profile, save_to_db):
        task_uuid = str(uuid.uuid4())
        db.create_pending_task(task_uuid, self.engine, profile, save_to_db)
        return task_uuid

    def _prepare_args(self, task_uuid, profile):
        raise NotImplementedError()

    def run(self, task_uuid, profile):
        args = self._prepare_args(task_uuid, profile)

        try:
            from subprocess import DEVNULL # py3k
        except ImportError:
            DEVNULL = open(os.devnull, 'wb')

        # run scan
        process = subprocess.Popen(args, stdout=DEVNULL)
        pid = process.pid

        # change db state from pending to in progess
        dt = datetime.datetime.now()
        db.set_in_progress_info(task_uuid=task_uuid, pid=pid, scan_start_time=dt)

        # wait process and process results
        process.wait()
        self._process_results(task_uuid)
        
    def _process_results(self, task_uuid):
        task_info = db.get_task_info(task_uuid)
        if not task_info['save_to_db']:
            db.set_results(task_uuid=task_uuid, results=None, state=next_state)
            return

        # determine next state
        if task_info['state'] == STATE_CANCELED:
            if not self.process_results_on_cancel:
                return
            else:
                next_state = STATE_CANCELED
        else:
            next_state = STATE_FINISHED

        tmp_file_path = os.path.join(self.tmp_dir_path, task_uuid)

        # task may be killed at that moment
        # and output file is empty or does not exist

        try:
            with open(tmp_file_path, 'r') as f:
                results, timestamps = self._parse_output(f.read())
                db.set_results(task_uuid=task_uuid, results=json.dumps(results), state=next_state,
                    scan_start_time=datetime.datetime.fromtimestamp(timestamps['scan_start']), 
                    scan_stop_time=datetime.datetime.fromtimestamp(timestamps['scan_stop']))
                
            os.remove(tmp_file_path)
        except IOError:
            db.set_results(task_uuid=task_uuid, results=None, state=next_state)

    def _parse_output(self, data):
        raise NotImplementedError()

    def cancel_scan(self, task_uuid):

        # get results from db by task_uuid
        task_info = db.get_task_info(task_uuid)
        # task_uuid not found
        if task_info is None:
            return {'status':'error', 'message':'unknown task_uuid'}

        # is pending and inprogress conditions
        cond11 = task_info['state'] == STATE_IN_PROGRESS
        cond12 = utils.is_process_alive(task_info['pid']) is True
        cond1 = cond11 and cond12
        cond2 = task_info['state'] == STATE_PENDING

        if cond1 or cond2:
            db.cancel_task(task_uuid)
            p = psutil.Process(task_info['pid']).kill()
            return {'status':'ok', 'message':'cancaled'}
        else:
            return {'status':'ok', 'message':'scan was already finished'}

    def get_info(self, task_uuid):
        # get results from db by task_uuid
        task_info = db.get_task_info(task_uuid)
        if task_info is None:
            return {'status':'error', 'message':'unknown task_uuid'}
        else:
            if task_info['results']:
                task_info['results'] = json.loads(task_info['results'])
            del task_info['pid']
            return {'status':'ok', 'task_info':task_info}
