from __future__ import unicode_literals, absolute_import, print_function

import shlex
import os
import subprocess
import datetime
import json
import sys
from xml.dom import minidom
from xml.parsers.expat import ExpatError
from future.utils import iteritems

from .base_engine import BaseEngine

from app import db
from app.settings import STATE_CANCELED, STATE_FINISHED
from app.settings import ENGINE_IVRE
from app.settings import IVRE_TEMPLATES_DIR


class IvreEngine(BaseEngine):
    def __init__(self):
        self.engine = ENGINE_IVRE
        self.process_results_on_cancel = True

    def _make_template(self, task_uuid, profile_):
        profile = profile_.copy()
        del profile["targets"]

        if not profile.get("nmap"):
            profile["nmap"] = "nmap"
        if not profile.get("pings"):
            profile["pings"] = ""
        if not profile.get("scans"):
            profile["scans"] = "S"
        if not profile.get("osdetect"):
            profile["osdetect"] = False
        if not profile.get("traceroute"):
            profile["traceroute"] = False
        if not profile.get("resolve"):
            profile["resolve"] = 1
        if not profile.get("verbosity"):
            profile["verbosity"] = 0
        if not profile.get("ports"):
            profile["ports"] = "-"
        if not profile.get("host_timeout"):
            profile["host_timeout"] = None
        if not profile.get("script_timeout"):
            profile["script_timeout"] = None
        if not profile.get("scripts_categories"):
            profile["scripts_categories"] = None
        if not profile.get("scripts_exclude"):
            profile["scripts_exclude"] = None
        if not profile.get("scripts_force"):
            profile["scripts_force"] = None
        if not profile.get("extra_options"):
            profile["extra_options"] = None

        template = 'NMAP_SCAN_TEMPLATES["custom"] = NMAP_SCAN_TEMPLATES["default"].copy()\n'
        template += 'NMAP_SCAN_TEMPLATES["custom"].update({})'.format(str(profile))

        self.template_path = os.path.join(IVRE_TEMPLATES_DIR, task_uuid)
        f = open(self.template_path, 'w')
        f.write(template)
        f.close()

    def _prepare_args(self, task_uuid, profile):
        self._make_template(task_uuid, profile)
        targets = profile['targets']
        return [self.engine, 'runscans', '--nmap-template', 'custom', '--categories', task_uuid] + shlex.split(targets)

    def run(self, task_uuid, profile):
        args = self._prepare_args(task_uuid, profile)

        # set env variable for template file
        custom_env = os.environ.copy()
        custom_env["IVRE_CONF"] = self.template_path

        # run scan
        process = subprocess.Popen(args, env=custom_env)
        pid = process.pid

        # change db state from pending to in progess
        dt = datetime.datetime.now()
        db.set_in_progress_info(task_uuid=task_uuid, pid=pid, scan_start_time=dt)

        # wait process and process results
        process.wait()
        self._process_results(task_uuid)

    def _process_results(self, task_uuid):
        task_info = db.get_task_info(task_uuid)
        if not task_info['save_to_db']:
            db.set_results(task_uuid=task_uuid, results=None, state=STATE_FINISHED)
            return

        # determine next state
        if db.get_task_info(task_uuid)['state'] == STATE_CANCELED:
            if not self.process_results_on_cancel:
                return
            else:
                next_state = STATE_CANCELED
        else:
            next_state = STATE_FINISHED

        self._run_scan2db(task_uuid)

        db.set_results(task_uuid=task_uuid, results=None, state=STATE_FINISHED)
        
    def _run_scan2db(self, task_uuid):
        out_dir = os.path.join(os.getcwd(), 'scans', task_uuid, 'up')

        # BECAUSE SOURCE FIELD IS 32 CHAR LENGTH
        ivre_scan_src = task_uuid.replace('-', '')

        args = [self.engine, 'scan2db', '-s', ivre_scan_src, '-r', out_dir]
        _return_code = subprocess.call(args)

    def get_info(self, task_uuid):
        task_info = db.get_task_info(task_uuid)
        if task_info is None:
            return {'status':'error', 'message':'unknown task_uuid'}
        else:
            if task_info['state'] in [STATE_FINISHED, STATE_CANCELED]:
                args = ['ivre', 'scancli', '--source', task_uuid.replace('-', ''), '--json']
                p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                output, _ = p.communicate()

                try:
                    task_info['results'] = json.loads(output)
                except ValueError:
                    task_info['results'] = None

            del task_info['pid']
            return {'status':'ok', 'task_info':task_info}