from __future__ import unicode_literals, absolute_import, print_function

import shlex
import os
import subprocess
import datetime
import json
from xml.dom import minidom
from xml.parsers.expat import ExpatError
from future.utils import iteritems

from .base_engine import BaseEngine

from app import db
from app.settings import TMP_DIR_PATH
from app.settings import STATE_CANCELED, STATE_FINISHED
from app.settings import ENGINE_MASSCAN
from app.settings import STATE_PORT_OPEN
from app.settings import STATE_HOST_UP


class MasscanEngine(BaseEngine):
    def __init__(self):
        self.engine = ENGINE_MASSCAN
        self.tmp_dir_path = TMP_DIR_PATH
        self.process_results_on_cancel = True

    def _prepare_args(self, task_uuid, profile):
        tmp_file_path = os.path.join(self.tmp_dir_path, task_uuid)
        args = [self.engine, '-oL', tmp_file_path] + shlex.split(profile['args'])
        return args

    def _parse_output(self, data_lines):
        res = []
        timestamps = {'scan_start': 0, 'scan_stop': 0}
        hosts = {}

        try:
            # first and last lines are comments
            # <status> <proto> <port> <ip> <timestamp>
            # <status> = open always???
            # <proto> actually only tcp
            for line in data_lines.splitlines()[1:-1]:

                # fields
                data = line.split(' ')
                portid = int(data[2])
                addr = data[3]
                response_ts = int(data[4])

                port_info = {'portid':portid, 'timestamp':response_ts}

                if not hosts.get(addr):
                    hosts[addr] = []
                hosts[addr].append(port_info)

                # update timestamps
                cond1 = timestamps['scan_start'] == 0
                cond2 = response_ts < timestamps['scan_start']
                if cond1 or cond2:
                    timestamps['scan_start'] = response_ts

                cond1 = timestamps['scan_stop'] == 0
                cond2 = response_ts > timestamps['scan_stop']
                if cond1 or cond2:
                    timestamps['scan_stop'] = response_ts

        except IndexError:
            # for the case when output finished incorrectly
            # cause of process killing on scan canceling
            pass

        # convert collected data to approproate format
        # Python 2, 3 comapible iteration over dict
        for (addr, ports) in iteritems(hosts):
            host_data = {'addr':addr, 'state':STATE_HOST_UP, 'timestamp':None}
            ports_data = []
            for port in ports:
                ports_data.append({'portid':port['portid'], 'state':STATE_PORT_OPEN,
                    'timestamp':port['timestamp'], 'protocol':None,
                    'service':None, 'product':None})

            res.append({'host':host_data, 'ports':ports_data})


        return res, timestamps
        