from __future__ import unicode_literals, absolute_import, print_function

import shlex
import os
import subprocess
import datetime
import json
import time
import sys
import traceback

from xml.dom import minidom
from xml.parsers.expat import ExpatError
import xml.etree.ElementTree as ET


from .base_engine import BaseEngine
from app import db
from app.utils import local_ts_2_msk_ts
from app.settings import TMP_DIR_PATH
from app.settings import STATE_CANCELED, STATE_FINISHED
from app.settings import ENGINE_NMAP


class NmapEngine(BaseEngine):
    def __init__(self):
        self.engine = ENGINE_NMAP
        self.tmp_dir_path = TMP_DIR_PATH
        self.process_results_on_cancel = False

    def _prepare_args(self, task_uuid, profile):
        tmp_file_path = os.path.join(self.tmp_dir_path, task_uuid)
        args = [self.engine, '-oX', tmp_file_path] + shlex.split(profile['args'])
        return args

    def _process_results(self, task_uuid):
        task_info = db.get_task_info(task_uuid)
        if not task_info:
            print('[+] nmap_engine. process_results. db.get_task_info(task_uuid) returned None. task_uuid: {}'.format(task_uuid))
        else:            
            db.set_results(task_uuid=task_uuid, results=None, state=STATE_FINISHED)
        return

    def prepare_results_from_xml(self, results):
        target_details = list()
        nmaprun = ET.fromstring(results)
        hosts = nmaprun.findall('host')

        for host in hosts:

            target = dict()
            target['addr'] = host.find('address').attrib['addr']
            target['starttime'] = int(host.attrib['starttime'])

            # OS DETECT
            os_matches = list()
            os = host.find('os')
            if os is not None:

                matches_ = os.findall('osmatch')
                if matches_ is not None:

                    for match in matches_:
                        if match.attrib is not None:
                            
                            name = match.attrib.get('name')
                            accuracy = match.attrib.get('accuracy')

                            if (name is not None) and (accuracy is not None):
                                os_matches.append({'name':name, 'accuracy':accuracy})

            target['os_matches'] = os_matches

            # PORTS
            target['ports'] = list()
            ports_ = host.find('ports')
            if ports_ is not None:
                for port in ports_.findall('port'):
                    port_details = dict()

                    # if port.find('state').attrib.get('state', 'unknown') != 'open':
                    #     continue

                    port_details['state'] = port.find('state').attrib.get('state', 'UNKNOWN')
                    port_details['port'] = int(port.attrib['portid'])
                    port_details['protocol'] = port.attrib['protocol']

                    ts = port.attrib.get('timestamp')
                    if ts:
                        ts = int(ts)
                    port_details['time'] = ts

                    # scripts
                    scripts_dict = dict()
                    for script in port.findall("script"):
                        script_json = script.attrib
                        scripts_dict[script_json['id']] = script_json['output']
                    port_details['scripts'] = scripts_dict

                    # service
                    service = port.find("service")
                    if (service is not None) and (service.attrib is not None) and service.attrib.get('method') == 'probed':
                        port_details['service_name'] = service.attrib.get('name')
                        port_details['service_product'] = service.attrib.get('product')
                        port_details['service_version'] = service.attrib.get('version')
                    else:
                        port_details['service_name'] = None
                        port_details['service_product'] = None
                        port_details['service_version'] = None

                    target['ports'].append(port_details)

            target_details.append(target)

        return target_details

    def get_info(self, task_uuid):
        task_info = db.get_task_info(task_uuid)

        if task_info is None:
            return {'status':'error', 'message':'unknown task_uuid'}
        
        if task_info['state'] in [STATE_FINISHED, STATE_CANCELED]:

            try:
                tmp_file_path = os.path.join(self.tmp_dir_path, task_uuid)
                f = open(tmp_file_path, 'r')
                results = f.read()
                f.close()
                os.remove(tmp_file_path)
                task_info['results'] = self.prepare_results_from_xml(results)

            except IOError:
                task_info['results'] = None
            except Exception:
                traceback.print_exc()
                print(results)
                task_info['results'] = None

        del task_info['pid']
        return {'status':'ok', 'task_info':task_info}
