from __future__ import unicode_literals, absolute_import, print_function

import shlex
import os
import subprocess
import datetime
import json
from xml.dom import minidom
from xml.parsers.expat import ExpatError

from .base_engine import BaseEngine

from app import db
from app.settings import TMP_DIR_PATH
from app.settings import ENGINE_ZMAP
from app.settings import STATE_PORT_OPEN
from app.settings import STATE_HOST_UP
from app.settings import STATE_CANCELED, STATE_FINISHED


METHOD_SYN_ACK = 1
METHOD_ICMP_ECHO = 2

class ZmapEngine(BaseEngine):
    def __init__(self):
        self.engine = ENGINE_ZMAP
        self.tmp_dir_path = TMP_DIR_PATH
        self.method = METHOD_SYN_ACK
        self.process_results_on_cancel = True

    def _determine_probe_method(self, args):
        """
        only icmp_echoscan and tcp_syn supported
        tcp_syn is default
        """
        cond1 = '--probe-module=icmp_echoscan' in args
        cond2 = '-M icmp_echoscan' in args
        if cond1 or cond2:
            self.method = METHOD_ICMP_ECHO
        else:
            self.method = METHOD_SYN_ACK

    def _prepare_args(self, task_uuid, profile):
        tmp_file_path = os.path.join(self.tmp_dir_path, task_uuid)

        self._determine_probe_method(profile['args'])
        if self.method == METHOD_SYN_ACK:
            output_args = '-O csv -o {} -f "saddr, sport, timestamp_ts"'.format(tmp_file_path)
        else:
            output_args = '-O csv -o {} -f "saddr, timestamp_ts"'.format(tmp_file_path)

        args = [self.engine] + shlex.split(output_args) + shlex.split(profile['args'])
        return args

    def _parse_output(self, data):
        if self.method == METHOD_SYN_ACK:
            return self._parse_syn_output(data)
        else:
            return self._parse_icmp_echo_output(data)

    def _parse_syn_output(self, data_lines):
        res = []
        timestamps = {'scan_start': 0, 'scan_stop': 0}

        try:
            # the first line if for field names, and not values
            for line in data_lines.splitlines()[1:]:

                # fields
                data = line.split(',')
                addr = data[0]
                portid = int(data[1])
                response_ts = int(data[2])

                # append host data and port data to results
                host_data = {'addr':addr, 'state':STATE_HOST_UP, 'timestamp':None}
                ports_data = [{'portid':portid, 'state':STATE_PORT_OPEN,
                    'timestamp':response_ts, 'protocol':None,
                    'service':None, 'product':None}]
                res.append({'host':host_data, 'ports':ports_data})

                # update timestamps
                cond1 = timestamps['scan_start'] == 0
                cond2 = response_ts < timestamps['scan_start']
                if cond1 or cond2:
                    timestamps['scan_start'] = response_ts

                cond1 = timestamps['scan_stop'] == 0
                cond2 = response_ts > timestamps['scan_stop']
                if cond1 or cond2:
                    timestamps['scan_stop'] = response_ts

        except IndexError:
            # for the case when output finished incorrectly
            # cause of process killing on scan canceling
            pass

        return res, timestamps

    def _parse_icmp_echo_output(self, data_lines):
        res = []
        timestamps = {'scan_start': 0, 'scan_stop': 0}

        try:
            # the first line if for field names, and not values
            for line in data_lines.splitlines()[1:]:

                # fields
                data = line.split(',')
                addr = data[0]
                response_ts = int(data[1])

                # append host data and port data to results
                host_data = {'addr':addr, 'state':STATE_HOST_UP, 'timestamp':response_ts}
                ports_data = []
                res.append({'host':host_data, 'ports':ports_data})

                # update timestamps
                cond1 = timestamps['scan_start'] == 0
                cond2 = response_ts < timestamps['scan_start']
                if cond1 or cond2:
                    timestamps['scan_start'] = response_ts

                cond1 = timestamps['scan_stop'] == 0 
                cond2 = response_ts > timestamps['scan_stop']
                if cond1 or cond2:
                    timestamps['scan_stop'] = response_ts

        except IndexError:
            # for the case when output finished incorrectly
            # cause of process killing on scan canceling
            pass

        return res, timestamps
