from __future__ import unicode_literals, absolute_import, print_function

import peewee as pw
import datetime, time
import functools

from .settings import STATE_PENDING, STATE_IN_PROGRESS
from .settings import STATE_CANCELED, STATE_FINISHED


db = pw.SqliteDatabase('taskinfo.db')

class AuthToken(pw.Model):
    value = pw.CharField()
    create_time = pw.DateTimeField(default=datetime.datetime.now)

    class Meta:
        database = db 

class TaskInfo(pw.Model):
    task_uuid = pw.UUIDField(primary_key=True)  # str
    pid = pw.IntegerField(null=True)            # int
    results = pw.CharField(null=True)           # str
    engine = pw.CharField()                     # str
    profile = pw.CharField()                    # str
    state = pw.CharField()                      # str (pending / in progress / cancaled / finished)
    save_to_db = pw.BooleanField()
    entry_create_time = pw.DateTimeField(default=datetime.datetime.now)   # datetime
    entry_modify_time = pw.DateTimeField(default=datetime.datetime.now)   # datetime
    scan_start_time = pw.DateTimeField(null=True)  # datetime
    scan_stop_time = pw.DateTimeField(null=True)   # datetime

    class Meta:
        database = db # This model uses the "taskinfo.db" databas

    def datetime_to_timestamp_or_none(self, dt):
        """
        if dt is None: return None
        else convert datetime to timestamp
        """
        return int(time.mktime(dt.timetuple())) if dt else None

    def as_dict(self):
        """
        returns dict representation of current Model
        """
        return {
            'task_uuid': self.task_uuid,
            'pid': self.pid,
            'results': self.results,
            'engine': self.engine,
            'profile': self.profile,
            'state': self.state,
            'save_to_db': self.save_to_db,
            'entry_create_time': self.datetime_to_timestamp_or_none(self.entry_create_time),
            'entry_modify_time': self.datetime_to_timestamp_or_none(self.entry_modify_time),
            'scan_start_time': self.datetime_to_timestamp_or_none(self.scan_start_time),
            'scan_stop_time': self.datetime_to_timestamp_or_none(self.scan_stop_time),
        }


# -----------------
# --- Decorator ---
# -----------------

def db_retries(func):

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        retries = 3
        while retries > 0:
            retries -= 1
            try:
                return func(*args, **kwargs)
            except pw.OperationalError as e:
                if retries == 0:
                    raise e
        return None

    return wrapper

# -------------
# --- Utils ---
# -------------

@db_retries
def create_pending_task(task_uuid, engine, profile, save_to_db):
    """
    Input:
        (str)task_uuid - uuid formated string
        (str)engine
        (str)profile
    """
    try:
        return TaskInfo.create(
            save_to_db=save_to_db, task_uuid=task_uuid, engine=engine, 
            profile=profile, state=STATE_PENDING
            ).as_dict()
    except pw.IntegrityError:
        return None


@db_retries
def set_in_progress_info(task_uuid, pid, scan_start_time):
    """
    Input:
        (str)task_uuid - uuid formated string
        (int)pid
        (datetime)scan_start_time
    """
    try:
        cur_dt = datetime.datetime.now()
        task = TaskInfo.get(TaskInfo.task_uuid == task_uuid)
        task.pid = pid
        task.scan_start_time = scan_start_time
        task.entry_modify_time = cur_dt
        task.state = STATE_IN_PROGRESS
        task.save()
        return True
    except pw.DoesNotExist:
        return False


@db_retries
def cancel_task(task_uuid):
    """
    Input:
        (str)task_uuid
    """
    try:
        cur_dt = datetime.datetime.now()
        task = TaskInfo.get(TaskInfo.task_uuid == task_uuid)
        task.scan_stop_time = cur_dt
        task.entry_modify_time = cur_dt
        task.state = STATE_CANCELED
        task.save()
        return True
    except pw.DoesNotExist:
        return False


@db_retries
def set_results(task_uuid, results, state=STATE_FINISHED, scan_start_time=None, scan_stop_time=None):
    """
    Input:
        (str)task_uuid  - uuid formated string
        (int)results    - json formated string
        (datetime / None)scan_start_time   - update start time if presents
        (datetime / None)scan_stop_time    - set current time if not specified other
    """
    try:
        cur_dt = datetime.datetime.now()
        task = TaskInfo.get(TaskInfo.task_uuid == task_uuid)

        if scan_start_time:
            task.scan_start_time = scan_start_time
        task.scan_stop_time = scan_stop_time if scan_stop_time else cur_dt

        task.entry_modify_time = cur_dt
        task.results = results
        task.state = state
        
        task.save()
        return True
    except pw.DoesNotExist:
        return False


@db_retries
def get_task_info(task_uuid):
    try:
        return TaskInfo.get(TaskInfo.task_uuid == task_uuid).as_dict()
    except pw.DoesNotExist:
        return None


@db_retries
def delete_task(task_uuid):
    try:
        TaskInfo.get(TaskInfo.task_uuid == task_uuid).delete_instance()
        return True
    except pw.DoesNotExist:
        return False


db.create_tables([TaskInfo, AuthToken], safe=True)
