#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import json
import base64
import requests
import argparse
import ipaddress
import traceback


DEFAULT_USER = 'default'
DEFAULT_READONLY_USER = 'readonly'
DEFAULT_PASS = ''
NOT_REAL_USER = 'audifge09fijkewnfoi2443foi3hq4pgnuh4h38h4iowah4oi3i3h4oh24rhg3u4'
CHECK_QUERY = '''EXISTS TABLE system.tables'''
VERSION_QUERY = '''SELECT version()'''


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("-t", "--targets", 
                        help='list of target-port pairs: "localhost:8123,127.0.0.1:8122,localhost"')
    parser.add_argument("-o", "--output", help='Output file name')
    return parser.parse_args()


def prepare_host(addr, port=8123):
    decoded_addr = addr.decode()
    prepare_addr = addr
    
    try:
        if ipaddress.ip_address(decoded_addr).version == 6:
            prepare_addr = '[{}]'.format(addr)
    except ValueError:
        pass

    return 'http://{}:{}'.format(prepare_addr, str(port))


def try_version(host, user, passwd):
    try:
        r = requests.post(host, auth=(user, passwd), params={'query': VERSION_QUERY}, timeout=3)
    except requests.ConnectionError:
        return None
    except requests.ReadTimeout:
        return None

    if 'DB::Exception' in r.text:
        return False
    else:
        return r.text[:-1]


def try_clickhouse(host, user, passwd):
    try:
        r = requests.post(host, auth=(user, passwd), params={'query': CHECK_QUERY}, timeout=3)
    except requests.ConnectionError:
        return None
    except requests.ReadTimeout:
        return None

    if 'DB::Exception' in r.text:
        return False
    elif r.text == '1\n':
        return True
    else:
        return None


def check_all(targets):
    results = list()
    for target in targets:

        target_info = {
            'target': target[0][1:-1] if target[0][0] == '[' and target[0][-1] == ']' else target[0],
            'port': target[1],
            'enabled': False,
            'default_auth': None,
            'readonly_auth': None,
            'version': None
        }

        prepared_host = prepare_host(target[0], target[1])

        res = try_clickhouse(prepared_host, DEFAULT_USER, DEFAULT_PASS)
        if res is not None:
            target_info['enabled'] = True
            target_info['default_auth'] = res
            target_info['version'] = try_version(prepared_host, DEFAULT_USER, DEFAULT_PASS)

        res = try_clickhouse(prepared_host, DEFAULT_READONLY_USER, DEFAULT_PASS)
        if res is not None:
            target_info['enabled'] = True
            target_info['readonly_auth'] = res
            if not target_info['version']:
                target_info['version'] = try_version(prepared_host, DEFAULT_READONLY_USER, DEFAULT_PASS)

        results.append(target_info)

    return results


def prepare_targets(targets):
    results = list()

    elems = targets.split(',')
    for elem in elems:
        stripped = elem.strip()
        parts = stripped.rsplit(':', 1)
        host = parts[0]
        port = int(parts[-1]) if len(parts) == 2 else 8123
        results.append((host, port))

    return results


def main():
    args = parse_args()

    targets = prepare_targets(args.targets)
    results = check_all(targets)

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()
