# -*- coding: utf-8 -*-
#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import re
import time
import json
import requests
import argparse
import traceback


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("-l", "--login", help='Login of user to act on behalf of', default="")
    parser.add_argument("-p", "--password", help='Password of user to act on behalf of', default="")
    parser.add_argument("-o", "--output", help='Output file name')
    return parser.parse_args()


def auth(login, password):
    s = requests.Session()
    data = {
        "login": login,
        "passwd": password,
        "twoweeks": "yes",
    }
    r = s.post("https://passport.yandex.ru/auth", data=data)
    ok = r.status_code == 200
    return s, ok


def check(login, password, required_iframe_url="https://trust.yandex.ru/web/binding"):
    results = {
        "host": "direct.yandex.ru",
        "url": "https://direct.yandex.ru",
        "actual_iframe_url": None,
        "success_auth": False,
        "trusted_iframe": False,
        "error": None,
        "required_iframe_url": required_iframe_url,
    }

    try:
        s, ok = auth(login, password)
        if not ok:
            results["error"] = "auth_request_failed"
            return results
        results["error"] = True
    
        session_id_cookie = s.cookies.get("Session_id")
        if not session_id_cookie:
            results["error"] = "no_session_id_cookie_set"
            return results

        url = "https://direct.yandex.ru/registered/main.pl?cmd=clientWallet&ulogin=&csrf_token="
        resp = s.get(url)
        if not (resp.status_code == 200):
            results["error"] = "direct_cmd_clientWallet_error"
            return results

        m = re.search(r"href=\"(https://direct\.yandex\.ru/registered/main\.pl\?cmd=autopaySettings.*?)\".*?</span></a>", resp.text)
        if not (m is not None):
            results["error"] = "cmd_autopaySettings_not_found_error"
            return results

        url = m.group(1)
        resp = s.get(url)
        if not (resp.status_code == 200):
            results["error"] = "direct_cmd_autopaySettings_error"
            return results

        m = re.search(r'\"url\":\"(https://direct\.yandex\.ru/registered/main\.pl\?cmd=ajaxGetBindingForm&amp.*?)\"', resp.text)
        if not (m is not None):
            results["error"] = "cmd_ajaxGetBindingForm_not_found_error"
            return results

        url = m.group(1)
        resp = s.get(url, allow_redirects=False)
        if not (resp.status_code >= 300 and resp.status_code < 400):
            results["error"] = "not_redirect_to_trust"
            return results

        location_header = resp.headers.get("Location")
        m = re.match(r"^([^\?]*).*$", location_header)
        if not m:
            results["error"] = "strange_location_format"
            return results

        location_url = m.group(1)
        results["actual_iframe_url"] = location_url

        if r"%D0%9D%D0%B5%20%D1%83%D0%B4%D0%B0%D0%BB%D0%BE%D1%81%D1%8C%20%D0%BF%D1" in location_header:
            results["error"] = "card_binding_error"
            return results

        if location_url != required_iframe_url:
            results["error"] = "all_is_fine_but_not_trusted_iframe"
            return results

        results["trusted_iframe"] = True

    except Exception as e:
        # embed()
        results["error"] = repr(e) + ": " + e.msg

    return results


def main():
    args = parse_args()

    result = None
    for i in range(5):
        result = check(args.login, args.password)
        if result["trusted_iframe"]:
            break
        time.sleep(10)

    results = [result]

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()
